/*******************************************************************************
 * Copyright (C) 2005 Chris Miles
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 ******************************************************************************/
package org.bbweather;

import java.io.ByteArrayInputStream;
import java.util.Hashtable;

import net.rim.device.api.util.Arrays;
import net.rim.device.api.xml.parsers.DocumentBuilder;
import net.rim.device.api.xml.parsers.DocumentBuilderFactory;

import org.bbweather.data.BarometricData;
import org.bbweather.data.CurrentConditionsData;
import org.bbweather.data.DayData;
import org.bbweather.data.ForecastData;
import org.bbweather.data.LocationData;
import org.bbweather.data.MoonData;
import org.bbweather.data.PartData;
import org.bbweather.data.UVIndexData;
import org.bbweather.data.UnitData;
import org.bbweather.data.WindData;
import org.bbweather.utils.IconUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import java.util.Date;
import net.rim.device.api.i18n.SimpleDateFormat;

public class WeatherDataManager {

    private static WeatherDataManager manager = null;

    private static Hashtable currentConditionsCache = new Hashtable();

    private static Hashtable forecastCache = new Hashtable();
    
    private static Hashtable hourlyCache = new Hashtable();

    public static WeatherDataManager getInstance() {

        if (manager == null)
            manager = new WeatherDataManager();

        return manager;
    }

    private WeatherDataManager() {
    }

    public CurrentConditionsData getCurrentConditionsByLocationId(String locationId) {

        CurrentConditionsData ccData = null;

        synchronized (currentConditionsCache) {
            ccData = (CurrentConditionsData) currentConditionsCache.get(locationId);
        }

        return ccData;
    }

    public ForecastData getForecastByLocationId(String locationId) {

        ForecastData forecastData = null;

        synchronized (forecastCache) {
            forecastData = (ForecastData) forecastCache.get(locationId);
        }

        return forecastData;
    }

    public void updateCurrentConditions(String locationId, String xml)
            throws Exception {

        Document document = weatherDataXML2Document(xml);
        Element root = document.getDocumentElement();

        // Check for error first
        //System.out.println("*** root node=" + root.getNodeName());
        if (root.getNodeName().equals("error")) {
            String errMsg = root.getFirstChild().getFirstChild().getNodeValue();
            throw new Exception(errMsg);
        }

        CurrentConditionsData ccData = parseCurrentConditions(root);
        synchronized (currentConditionsCache) {
            currentConditionsCache.put(locationId, ccData);
        }
    }

    public void updateForecast(String locationId, String xml) throws Exception {

        Document document = weatherDataXML2Document(xml);
        Element root = document.getDocumentElement();

        // Check for error first
        //System.out.println("*** root node=" + root.getNodeName());
        if (root.getNodeName().equals("error")) {
            String errMsg = root.getFirstChild().getFirstChild().getNodeValue();
            throw new Exception(errMsg);
        }

        ForecastData forecastData = parseForecast(root);
        synchronized (forecastCache) {
            forecastCache.put(locationId, forecastData);
        }
    }
    
    public ForecastData getHourlyByLocationId(String locationId) {

        ForecastData hourlyData = null;

        synchronized (hourlyCache) {
            hourlyData = (ForecastData) hourlyCache.get(locationId);
        }

        return hourlyData;
    }
    
    public void updateHourly(String locationId, String xml) throws Exception {

        Document document = weatherDataXML2Document(xml);
        Element root = document.getDocumentElement();

        // Check for error first
        //System.out.println("*** root node=" + root.getNodeName());
        if (root.getNodeName().equals("error")) {
            String errMsg = root.getFirstChild().getFirstChild().getNodeValue();
            throw new Exception(errMsg);
        }

        ForecastData hourlyData = parseHourly(root);
        synchronized (hourlyCache) {
            hourlyCache.put(locationId, hourlyData);
        }
    }
    
    private ForecastData parseHourly(Element root) {

        ForecastData hourlyData = new ForecastData();

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** element name=" + element.getNodeName());
            if (element.getNodeName().equals("head")) {
                hourlyData.unitData = getUnitData(element);
            } else if (element.getNodeName().equals("loc")) {
                hourlyData.locationData = getLocationData(element);
            } else if (element.getNodeName().equals("hbhf")) {
                getHourlyData(element, hourlyData);
            }
        }

        return hourlyData;
    }
    
    private void getHourlyData(Element root, ForecastData hourlyData) {

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** hbhf element name=" + element.getNodeName());
            if (element.getNodeName().equals("lsup")) {
                hourlyData.lastUpdate = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("hour")) {
                DayData dayData = getHourData(element);
                Arrays.add(hourlyData.dayData, dayData);
                hourlyData.numberDays++;
            }
        }
    }
    
    private DayData getHourData(Element root) {

        DayData dayData = new DayData();
        dayData.dayPartData = new PartData();

        dayData.dayNumber = root.getAttribute("h");
        dayData.dayName = root.getAttribute("c");
        int dayNumber = Integer.parseInt(dayData.dayNumber);
        int theHour = Integer.parseInt(dayData.dayName);
        /*if (theHour == 0)
            dayData.dayName = "12:00a";
        else if (theHour == 12)
            dayData.dayName = "12:00p";
        else if (theHour < 10)
            dayData.dayName = "0" + theHour + ":00a";
        else if (theHour > 12 && theHour < 22)
            dayData.dayName = "0" + (theHour - 12) + ":00p";
        else if (theHour > 21)
            dayData.dayName = (theHour - 12) + ":00p";
        else
            dayData.dayName = theHour + ":00a";*/
            
        SimpleDateFormat sdFormat = new SimpleDateFormat("M/dd");
        String theTime;
        
        Date d = new Date();
        if (theHour < dayNumber)
            d.setTime(d.getTime() + 1000 * 60 * 60 * 24);

        theTime = sdFormat.formatLocal(d.getTime());
        //theTime = sdFormat.formatLocal(new Date().getTime());
                        
        if (theHour == 0)
            dayData.dayName = "12am";
        else if (theHour == 12)
            dayData.dayName = "12pm";
        else if (theHour < 12)
            dayData.dayName = theHour + "am";
        else
            dayData.dayName = (theHour - 12) + "pm";
        
        dayData.date = theTime;
        //dayData.date = dayData.dayName;

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** day element name=" + element.getNodeName());
            if (element.getNodeName().equals("tmp")) {
                dayData.hiTemp = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("flik")) {
                dayData.lowTemp = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("icon")) {
                dayData.dayPartData.iconNumber = element.getFirstChild().getNodeValue();
                dayData.dayPartData.bitmap = IconUtils.getBitmap("/images/weather/"
                                                    + dayData.dayPartData.iconNumber + ".png");
            } else if (element.getNodeName().equals("t")) {
                dayData.dayPartData.description = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("wind")) {
                dayData.dayPartData.windData = getWindData(element);
            } else if (element.getNodeName().equals("ppcp")) {
                dayData.dayPartData.precipitation = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("hmid")) {
                dayData.dayPartData.humidity = element.getFirstChild().getNodeValue();
            }
        }

        return dayData;
    }

    private Document weatherDataXML2Document(String xml) throws Exception {

        Document document = null;
        ByteArrayInputStream bis = null;

        try {
            bis = new ByteArrayInputStream(xml.getBytes());
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setIgnoringElementContentWhitespace(true);
            factory.setAllowUndefinedNamespaces(true);
            DocumentBuilder builder = factory.newDocumentBuilder();
            document = builder.parse(bis);
        } finally {
            if (bis != null)
                bis.close();
        }

        return document;
    }

    private CurrentConditionsData parseCurrentConditions(Element root) {

        CurrentConditionsData ccData = new CurrentConditionsData();

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** element name=" + element.getNodeName());
            if (element.getNodeName().equals("head")) {
                ccData.unitData = getUnitData(element);
            } else if (element.getNodeName().equals("loc")) {
                ccData.locationData = getLocationData(element);
            } else if (element.getNodeName().equals("cc")) {
                getCurrentConditionsData(element, ccData);
            }
        }

        return ccData;
    }

    private ForecastData parseForecast(Element root) {

        ForecastData forecastData = new ForecastData();

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** element name=" + element.getNodeName());
            if (element.getNodeName().equals("head")) {
                forecastData.unitData = getUnitData(element);
            } else if (element.getNodeName().equals("loc")) {
                forecastData.locationData = getLocationData(element);
            } else if (element.getNodeName().equals("dayf")) {
                getForecastData(element, forecastData);
            }
        }

        return forecastData;
    }

    private UnitData getUnitData(Element root) {

        UnitData unitData = new UnitData();

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** unit element name=" + element.getNodeName());
            if (element.getNodeName().equals("ut")) {
                unitData.temperature = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("ud")) {
                unitData.distance = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("us")) {
                unitData.speed = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("up")) {
                unitData.precipitation = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("ur")) {
                unitData.pressure = element.getFirstChild().getNodeValue();
            }
        }

        return unitData;
    }

    private LocationData getLocationData(Element root) {

        LocationData locationData = new LocationData();

        locationData.id = root.getAttribute("id");

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** location element name=" + element.getNodeName());
            if (element.getNodeName().equals("dnam")) {
                locationData.name = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("tm")) {
                locationData.lastUpdate = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("sunr")) {
                locationData.sunrise = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("suns")) {
                locationData.sunset = element.getFirstChild().getNodeValue();
            }
        }

        return locationData;
    }

    private void getForecastData(Element root, ForecastData forecastData) {

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** dayf element name=" + element.getNodeName());
            if (element.getNodeName().equals("lsup")) {
                forecastData.lastUpdate = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("day")) {
                DayData dayData = getDayData(element);
                Arrays.add(forecastData.dayData, dayData);
                forecastData.numberDays++;
            }
        }
    }

    private void getCurrentConditionsData(Element root,
                                          CurrentConditionsData ccData) {

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** cc element name=" + element.getNodeName());
            if (element.getNodeName().equals("lsup")) {
                ccData.lastUpdate = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("obst")) {
                ccData.observationStation = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("tmp")) {
                ccData.temperature = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("flik")) {
                ccData.feelsLikeTemperature = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("t")) {
                ccData.description = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("icon")) {
                ccData.iconNumber = element.getFirstChild().getNodeValue();
                ccData.bitmap = IconUtils.getBitmap("/images/weather/"
                        + ccData.iconNumber + ".png");
            } else if (element.getNodeName().equals("bar")) {
                ccData.barometricData = getBarometricData(element);
            } else if (element.getNodeName().equals("wind")) {
                ccData.windData = getWindData(element);
            } else if (element.getNodeName().equals("hmid")) {
                ccData.humidity = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("vis")) {
                ccData.visibility = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("uv")) {
                ccData.uvIndexData = getUVIndexData(element);
            } else if (element.getNodeName().equals("dewp")) {
                ccData.dewPoint = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("moon")) {
                ccData.moonData = getMoonData(element);
            }
        }
    }

    private BarometricData getBarometricData(Element root) {

        BarometricData barometricData = new BarometricData();

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** barometric element name=" + element.getNodeName());
            if (element.getNodeName().equals("r")) {
                barometricData.currentPressure = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("d")) {
                barometricData.description = element.getFirstChild().getNodeValue();
            }
        }

        return barometricData;
    }

    private WindData getWindData(Element root) {

        WindData windData = new WindData();

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** wind element name=" + element.getNodeName());
            if (element.getNodeName().equals("s")) {
                windData.speed = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("gust")) {
                windData.gustSpeed = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("d")) {
                windData.direction = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("t")) {
                windData.description = element.getFirstChild().getNodeValue();
            }
        }

        return windData;
    }

    private UVIndexData getUVIndexData(Element root) {

        UVIndexData uvData = new UVIndexData();

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** uv element name=" + element.getNodeName());
            if (element.getNodeName().equals("i")) {
                uvData.value = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("t")) {
                uvData.description = element.getFirstChild().getNodeValue();
            }
        }

        return uvData;
    }

    private MoonData getMoonData(Element root) {

        MoonData moonData = new MoonData();

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** moon element name=" + element.getNodeName());
            if (element.getNodeName().equals("icon")) {
                moonData.iconNumber = element.getFirstChild().getNodeValue();
                moonData.bitmap = IconUtils.getBitmap("/images/weather/"
                                                    + moonData.iconNumber + ".png");
            } else if (element.getNodeName().equals("t")) {
                moonData.description = element.getFirstChild().getNodeValue();
            }
        }

        return moonData;
    }

    private DayData getDayData(Element root) {

        DayData dayData = new DayData();

        dayData.dayNumber = root.getAttribute("d");
        dayData.dayName = root.getAttribute("t");
        dayData.date = root.getAttribute("dt");

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** day element name=" + element.getNodeName());
            if (element.getNodeName().equals("hi")) {
                dayData.hiTemp = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("low")) {
                dayData.lowTemp = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("sunr")) {
                dayData.sunrise = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("suns")) {
                dayData.sunset = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("part")) {
                String type = element.getAttribute("p");
                if (type.equals("d"))
                    dayData.dayPartData = getPartData(element);
                else
                    dayData.nightPartData = getPartData(element);
            }
        }

        return dayData;
    }

    private PartData getPartData(Element root) {

        PartData partData = new PartData();

        NodeList nodes = root.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Element element = (Element) nodes.item(i);
            //System.out.println("*** part element name=" + element.getNodeName());
            if (element.getNodeName().equals("icon")) {
                partData.iconNumber = element.getFirstChild().getNodeValue();
                partData.bitmap = IconUtils.getBitmap("/images/weather/"
                                                    + partData.iconNumber + ".png");
            } else if (element.getNodeName().equals("t")) {
                partData.description = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("wind")) {
                partData.windData = getWindData(element);
            } else if (element.getNodeName().equals("ppcp")) {
                partData.precipitation = element.getFirstChild().getNodeValue();
            } else if (element.getNodeName().equals("hmid")) {
                partData.humidity = element.getFirstChild().getNodeValue();
            }
        }

        return partData;
    }
}
