//this file is part of notepad++
//Copyright (C)2003 Don HO ( donho@altern.org )
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either
//version 2 of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include "PluginInterface.h"
#include "Scintilla.h"

#include <string>
#include <vector>
#include <algorithm>

using namespace std;

// This is the name which will be displayed in Plugins Menu
const char PLUGIN_NAME[] = "Sort Lines";

NppData nppData;

// The number of functionality
const int nbFunc = 2;
// Declaration of functionality (FuncItem) Array
FuncItem funcItem[nbFunc];


// Here're the declaration of functions for 2 functions
void Ascending();
void Descending();

void SortLines(bool bAscending);

int m_iEOLSize = 0;

bool SortAscending(const string& S1, const string& S2)
{
	//Compare how may chars? Use the length of the smaller string
	int mLength = S1.length() < S2.length() ? S1.length() : S2.length();

	//Don't compare the end of line chars
	mLength -= m_iEOLSize;

	if ( S1.length() <= m_iEOLSize )
		return TRUE; //An empty Line is always smaller
	else if ( S2.length() <= m_iEOLSize )
		return FALSE; //An empty Line is always smaller
	else if ( strnicmp(S1.c_str(), S2.c_str(), mLength) < 0 )
		return TRUE; //S1 is smaller
	else if ( strnicmp(S1.c_str(), S2.c_str(), mLength) == 0 )
	{
		//S1 and S2 are identical but possibly with diff case, "A" and "a" or diff size
		if ( S1.length() < S2.length() )
			return TRUE;
		else if ( strncmp(S1.c_str(), S2.c_str(), mLength) <= 0 )
			return FALSE; //S1 is upper and S2 is lower
		else
			return TRUE;
	}
	
	return FALSE;
}

bool SortDescending(const string& S1, const string& S2)
{
	//Compare how may chars? Use the length of the smaller string
	int mLength = S1.length() < S2.length() ? S1.length() : S2.length();

	//Don't compare the end of line chars
	mLength -= m_iEOLSize;

	if ( S1.length() <= m_iEOLSize )
		return FALSE; //An empty Line is always smaller
	else if ( S2.length() <= m_iEOLSize )
		return TRUE; //An empty Line is always smaller
	else if ( strnicmp(S1.c_str(), S2.c_str(), mLength) < 0 )
		return FALSE; //S1 is smaller
	else if ( strnicmp(S1.c_str(), S2.c_str(), mLength) == 0 )
	{
		//S1 and S2 are identical but possibly with diff case, "A" and "a" or diff size
		if ( S1.length() < S2.length() )
			return FALSE;
		else if ( strncmp(S1.c_str(), S2.c_str(), mLength) <= 0 )
			return TRUE; //S1 is upper and S2 is lower
		else
			return FALSE;
	}
	
	return TRUE;
}

BOOL APIENTRY DllMain( HANDLE hModule, DWORD reasonForCall, LPVOID lpReserved)
{
	switch (reasonForCall)
	{
		// Here we initialize the FuncItem Array.
		// AWARE : no need to initialize _cmdID field.
		// This filed will be initialized by Notepad++ plugins system
		case DLL_PROCESS_ATTACH:
			funcItem[0]._pFunc = Ascending;
			funcItem[1]._pFunc = Descending;

			strcpy(funcItem[0]._itemName, "Ascending");
			strcpy(funcItem[1]._itemName, "Descending");

			// if you want your menu item is checked while the launch of program
			// set it as true, otherwise set it as false
			funcItem[0]._init2Check = false;
			funcItem[1]._init2Check = false;
		break;

		case DLL_PROCESS_DETACH:
		break;

		case DLL_THREAD_ATTACH:
		break;

		case DLL_THREAD_DETACH:
		break;
}

return TRUE;
}

/*
 *--------------------------------------------------
 * The 4 extern functions are mandatory
 * They will be called by Notepad++ plugins system
 *--------------------------------------------------
*/

// The setInfo function gets the needed infos from Notepad++ plugins system
extern "C" __declspec(dllexport) void setInfo(NppData notpadPlusData)
{
	nppData = notpadPlusData;
}

// The getName function tells Notepad++ plugins system its name
extern "C" __declspec(dllexport) const char * getName()
{
	return PLUGIN_NAME;
}

// The getFuncsArray function gives Notepad++ plugins system the pointer FuncItem Array
// and the size of this array (the number of functions)
extern "C" __declspec(dllexport) FuncItem * getFuncsArray(int *nbF)
{
	*nbF = nbFunc;
	return funcItem;
}

// If you don't need get the notification from Notepad++,
// just let it be empty.
extern "C" __declspec(dllexport) void beNotified(SCNotification *notifyCode)
{

}

//----------------------------------------------------------//
// Here're the definition of functions for 3 functionalities
//----------------------------------------------------------//
void SortLines(bool bAscending)
{
	int m_iLineLength = 0, m_iStartPos = 0, m_iEndPos = 0, m_iEOL, m_iFirstLine = 0, m_iLastLine = 0;
	int m_iLineCount = 0;
	int currentEdit;
	vector<string> strLines;
	string strLine;
	static const basic_string <char>::size_type npos = -1;
	char *szLine, szEOL[3];

	HWND hCurrentEditView = nppData._scintillaMainHandle;
	::SendMessage(nppData._nppHandle, WM_GETCURRENTSCINTILLA, 0, (LPARAM)&currentEdit);
	if (currentEdit)
		hCurrentEditView = nppData._scintillaSecondHandle;

	m_iFirstLine = ::SendMessage(hCurrentEditView, SCI_GETANCHOR, 0, (LPARAM)0);
	m_iLastLine = ::SendMessage(hCurrentEditView, SCI_GETCURRENTPOS, 0, (LPARAM)0);
	if ( m_iLastLine < m_iFirstLine )
	{
		int m_iTemp = m_iLastLine;
		m_iLastLine = m_iFirstLine;
		m_iFirstLine = m_iTemp;
	}

	m_iFirstLine = ::SendMessage(hCurrentEditView, SCI_LINEFROMPOSITION, m_iFirstLine, 0);
	m_iLastLine = ::SendMessage(hCurrentEditView, SCI_LINEFROMPOSITION, m_iLastLine, 0);
	m_iLineCount = ::SendMessage(hCurrentEditView, SCI_GETLINECOUNT, 0, (LPARAM)0);

	if ( m_iFirstLine == m_iLastLine )
	{
		m_iFirstLine = 0;
		m_iLastLine = m_iLineCount - 1;
	}
	m_iEOL = ::SendMessage(hCurrentEditView, SCI_GETEOLMODE, 0, (LPARAM)0);

	if ( m_iEOL == SC_EOL_CRLF ) //windows
	{
		m_iEOLSize = 2;
		strcpy(szEOL, "\r\n");
	}
	else if ( m_iEOL == SC_EOL_CR ) //unix
	{
		m_iEOLSize = 1;
		strcpy(szEOL, "\n");
	}
	else //mac
	{
		m_iEOLSize = 1;
		strcpy(szEOL, "\r");
	}

	for (int i = m_iFirstLine; i <= m_iLastLine; i++)
	{
		m_iLineLength = ::SendMessage(hCurrentEditView,
			SCI_LINELENGTH, (WPARAM)i, (LPARAM)0);

		szLine = new char[m_iLineLength + 1];
		strLine.resize(m_iLineLength + 1);

		::SendMessage(hCurrentEditView,
			SCI_GETLINE, (WPARAM)i, (LPARAM)&szLine[0]);

		memset(&szLine[m_iLineLength], 0, 1);
		if (m_iLineLength > 0)
			strLine.assign(szLine);
		else
			strLine.assign(szEOL);

		if ( strLine.at(strLine.length() - 1) != '\r' && strLine.at(strLine.length() - 1) != '\n' )
			strLine.append(szEOL);

		strLines.push_back(strLine);
		delete [] szLine;
	}

	if (bAscending)
		sort(strLines.begin(), strLines.end(), SortAscending);
	else
		sort(strLines.begin(), strLines.end(), SortDescending);

	m_iStartPos = ::SendMessage(hCurrentEditView, SCI_POSITIONFROMLINE, m_iFirstLine, 0);
	m_iEndPos = ::SendMessage(hCurrentEditView, SCI_POSITIONFROMLINE, m_iLastLine+1, 0);

	::SendMessage(hCurrentEditView, SCI_BEGINUNDOACTION, 0, 0);
	::SendMessage(hCurrentEditView, SCI_SETTARGETSTART, m_iStartPos, 0);
	::SendMessage(hCurrentEditView, SCI_SETTARGETEND, m_iEndPos, 0);

	::SendMessage(hCurrentEditView, SCI_REPLACETARGET, 0, (LPARAM)"");
	//::SendMessage(hCurrentEditView, SCI_CLEARALL, (WPARAM)0, (LPARAM)0);

	for (int i = 0; i < strLines.size(); i++)
	{
		//If last item, erase new line characters
		if ( i + 1 == strLines.size()
			&& strLines[i].length() >= m_iEOLSize
			&& m_iLineCount == m_iLastLine - m_iFirstLine + 1 )
				strLines[i].erase(strLines[i].length() - m_iEOLSize, m_iEOLSize);

		::SendMessage(hCurrentEditView, SCI_ADDTEXT,
			(WPARAM)strLines[i].length(), (LPARAM)strLines[i].c_str());

	}

	::SendMessage(hCurrentEditView, SCI_ENDUNDOACTION, 0, 0);
}

void Ascending()
{
	SortLines(true);
}

void Descending()
{
	SortLines(false);
}