/*
 OTFExplorer.cpp
 main program

 Copyright (c) 2006 Josh Harris
	
 Redistribution and use in source and binary forms, with or without modification,
 are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice,
     this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright notice,
     this list of conditions and the following disclaimer in the documentation and/or
     other materials provided with the distribution.
  3. The name of the author may not be used to endorse or promote products derived
     from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE
 LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

	2006-10-18 - JHH
		* Original release
*/

//#define WIN32_LEAN_AND_MEAN		// Exclude rarely-used stuff from Windows headers
#define _WIN32_WINNT	0x0500
#include <windows.h>
#include <stdlib.h>
#include <stdio.h>
#include <tchar.h>
#include <iostream>
#include <vector>
#include <string>
#include <fcntl.h>

#include "crypto.h"
#include "./filesys/filesys.h"

using namespace std;

extern OTFE_CONTEXT *crypt_ctx;

////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
char *error_code(int err) {
	switch (err) {
		case EACCES:
			return "Invalid open flag chosen";
		case EBADF:
			return "Invalid state";
		case EEXIST:
			return "file or directory already exists";
		case EFBIG:
			return "EFBIG";
		case EINVAL:
			return "Invalid path selected";
		case EIO:
			return "invalid cluster or directory entry";
		case EISDIR:
			return "Cannot open a directory";
		case ENODEV:
			return "Partition not open";
		case ENOENT:
			return "File/Directory not found";
		case ENOSPC:
			return "Not enough space to add new";
		case ENOTDIR:
			return "path is not a valid directory";
		case ENOTSUP:
			return "ENOTSUP";
		case EOVERFLOW:
			return "Attempted to access past end of file";
		case EPERM:
			return "The directory is not empty";
		case EROFS:
			return "Partition is read only";
		case EXDEV:
			return "file/directory found on wrong partition";
		default:
			return "Unknown error";
	}
}

const char Usage[] = "OTFExplorer -v file.tc -p password [options]\n"
"  -s SourceFile\n"
"  -s DestinationFile\n"
"  -m method\n"
"    -m e (Export file 'source' from container to local drive 'dest')\n"
"    -m i (Import file 'source' from local drive to container 'dest)\n"
"    -m r (Rename a file or folder from 'source' to 'dest' in the container)\n"
"    -m f (Create a folder 'source' in the container)\n"
"    -m d (Delete a file/folder 'source' from the container)\n"
"    -m l (List all files in the container)\n"
"    -m s (Show container partition info)\n"
"\n"
"Examples:\n"
"  Export a file from the container\n"
"    -v \"file.tc\" -p 1234 -m e -s \"/path_to/file.avi\" -d \"c:\\file.avi\"\n"
"  Import a file into the container\n"
"    -v \"file.tc\" -p 1234 -m i -s \"c:\\file.avi\" -d \"/path_to/file.avi\"\n"
"  Create a folder\n"
"    -v \"file.tc\" -p 1234 -m f -s \"/path_to/new folder\"\n"
"  List all files in the container\n"
"    -v \"file.tc\" -p 1234 -m l\n";

int registerCrypt()
{
	if (register_cipher(&aes_desc) == -1) {
		printf("Error registering aes_desc\n");
		return -1;
	}
	if (register_cipher(&blowfish_desc) == -1) {
		printf("Error registering blowfish_desc\n");
		return -1;
	}
	if (register_cipher(&cast5_desc) == -1) {
		printf("Error registering cast5_desc\n");
		return -1;
	}
	if (register_cipher(&serpent_desc) == -1) {
		printf("Error registering serpent_desc\n");
		return -1;
	}
	if (register_cipher(&twofish_desc) == -1) {
		printf("Error registering twofish_desc\n");
		return -1;
	}
	if (register_cipher(&des3_desc) == -1) {
		printf("Error registering des3_desc\n");
		return -1;
	}


	if (register_hash(&rmd160_desc) == -1) {
		printf("Error registering RipeMD160\n");
		return -1;
	}
	if (register_hash(&sha1_desc) == -1) {
		printf("Error registering SHA1\n");
		return -1;
	}
	if (register_hash(&whirlpool_desc) == -1) {
		printf("Error registering Whirlpool\n");
		return -1;
	}

	if (register_prng(&fortuna_desc) == -1) {
		printf("Error registering Fortuna prng\n");
		return -1;
	}

	return 0;
}

void ListAll_Dirs(char *theDir)
{
	DIR_STATE_STRUCT mDirState;
	DIR_STATE_STRUCT *mDirNextState;
	memset(&mDirState, 0, sizeof(mDirState));
	char filename[MAX_PATH];
	char dirname[MAX_PATH];
	struct stat filestat;

	struct _reent r;

	int retVal;

	mDirNextState = _FAT_diropen_r(&r, &mDirState, theDir);
	if (mDirState.validEntry) {
		while (mDirState.validEntry) {
			if (_FAT_directory_isDirectory(&mDirState.currentEntry)) {
				if (mDirState.currentEntry.filename[0] != '.') {
					if (strcmp(theDir, "/") == 0)
						sprintf(dirname, "/%s", mDirState.currentEntry.filename);
					else
						sprintf(dirname, "%s/%s", theDir, mDirState.currentEntry.filename);

					//printf("Dir : %s\n", dirname);
					ListAll_Dirs(dirname);
				}
			} else {
				if (strcmp(theDir, "/") == 0)
					printf("File: /%s\n", mDirState.currentEntry.filename);
				else
					printf("File: %s/%s\n", theDir, mDirState.currentEntry.filename);
			}

			_FAT_dirnext_r(&r, &mDirState, filename, &filestat);
		}
		retVal = _FAT_dirclose_r(&r, &mDirState);
		retVal = _FAT_dirclose_r(&r, mDirNextState);
		retVal = 2;
	} else {
		printf("No files or directories found: %d\n%s\n", r._errno, error_code(r._errno));
	}
}

int _tmain(int argc, _TCHAR* argv[])
{
	HANDLE m_hVolume, m_hFile;
	DWORD dwVolAccess = GENERIC_READ | GENERIC_WRITE;
	DWORD dwBytesRead = 0;
	ULARGE_INTEGER m_i64FileSz;
	LARGE_INTEGER m_iPos;
	byte inbuf[512];
	byte headerCipher[1024], headerPlain[512];
	BOOL bSuccess = FALSE;

	dwBytesRead = VirtualLock(&headerPlain, 512);

	PARTITION *partition = NULL;
	FILE_STRUCT mFile;
	int fd, method = 0, flags = 0, mode = 0, retVal = 0;
	struct _reent r;

	dwBytesRead = sizeof(*partition);
	dwBytesRead = VirtualLock(&partition, sizeof(*partition));

	TCHAR *szVolume = NULL;
	TCHAR *szSource = NULL;
	char strSource[MAX_PATH];
	TCHAR *szDest = NULL;
	char strDest[MAX_PATH];
	TCHAR *szPwd = NULL;
	char strPwd[MAX_PATH];
	char *fileBuf;

	DWORD dwBytesWritten = 0;

	r._errno = 0;

	printf("OTFExplorer v0.1 Experimental\n"
		"Please do not use on any OTF containers with valuable data\n"
		"  that you can't afford to lose\n\n");

	if (argc < 2) {
		printf(Usage);
		return 1;
	}

	for (int i = 1; i+1 < argc; i++) {
		if (argv[i][0] == '-' && _tcslen(argv[i]) == 2) {
			switch (argv[i][1]) {
				case 'v':
					szVolume = argv[i+1];
					break;
				case 's':
					szSource = argv[i+1];
					break;
				case 'd':
					szDest = argv[i+1];
					break;
				case 'p':
					szPwd = argv[i+1];
					//wcstombs(strPwd, szPwd, sizeof(strPwd)); 
					break;
				case 'm':
					switch (argv[i+1][0]) {
						case 'e': //export file
							method = 0;
							dwVolAccess = GENERIC_READ;
							break;
						case 'i': //import file
							method = 1;
							break;
						case 'r': //rename file
							method = 2;
							break;
						case 'f': //Create folder
							method = 3;
							break;
						case 'd': //delete file/folder
							method = 4;
							break;
						case 'l': //list all files
							method = 5;
							dwVolAccess = GENERIC_READ;
							break;
						case 's': //Show partition info
							method = 6;
							dwVolAccess = GENERIC_READ;
							break;
						default:
							method = 0;
							break;
					}
					break;
				default:
					break;
			}
			i++;
		}
	}

	if (szVolume == NULL || szPwd == NULL
		(method < 3 && szSource == NULL && szDest == NULL) ||
		(method > 2 && method < 5 && szSource == NULL)) {
		printf(Usage);
		return 1;
	}

	char *pDest;
	if (szSource != NULL) {
		while(pDest = strchr(szSource, '\\')) {
			memset(&szSource[(int)(pDest - szSource)], '/', 1);
		}
	}
	if (szDest != NULL) {
		while(pDest = strchr(szDest, '\\')) {
			memset(&szDest[(int)(pDest - szDest)], '/', 1);
		}
	}

	if (registerCrypt() == -1) {
		if (szPwd)
			SecureZeroMemory(szPwd, strlen(szPwd));
		return 1;
	}

	m_hVolume = CreateFile(szVolume, dwVolAccess,
				FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
				OPEN_EXISTING, FILE_FLAG_NO_BUFFERING,
				NULL);

	if (m_hVolume == INVALID_HANDLE_VALUE) {
		LPVOID lpMsgBuf;

        FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
                      NULL,
                      GetLastError(),
                      MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
                      (LPTSTR) &lpMsgBuf,
                      0,
                      NULL);

		printf("Error loading volume\n%s\n%s\n", (LPCTSTR)lpMsgBuf, szVolume);

		LocalFree(lpMsgBuf);
		if (szPwd)
			SecureZeroMemory(szPwd, strlen(szPwd));
		return 1;
	}

	if (!ReadFile(m_hVolume, inbuf, 512, &dwBytesRead, FALSE)) {
		printf("Cannot read standard volume header\n");
		CloseHandle(m_hVolume);
		if (szPwd)
			SecureZeroMemory(szPwd, strlen(szPwd));
		return 1;
	}

	memcpy(headerCipher, inbuf, 512);

	m_i64FileSz.LowPart = GetFileSize(m_hVolume, &m_i64FileSz.HighPart);

	if (m_i64FileSz.QuadPart < 2048)
		m_iPos.QuadPart = -SECTOR_SIZE;
	else
		m_iPos.QuadPart = -3 * SECTOR_SIZE;

	if (!SetFilePointer(m_hVolume, m_iPos.LowPart, &m_iPos.HighPart, FILE_END)) {
		printf("Cannot seek to hidden volume header position\n");
		CloseHandle(m_hVolume);
		if (szPwd)
			SecureZeroMemory(szPwd, strlen(szPwd));
		return 1;
	}

	if (!ReadFile(m_hVolume, inbuf, 512, &dwBytesRead, FALSE)) {
		printf("Cannot read hidden volume header\n");
		CloseHandle(m_hVolume);
		if (szPwd)
			SecureZeroMemory(szPwd, strlen(szPwd));
		return 1;
	}

	memcpy(&headerCipher[512], inbuf, 512);

	bSuccess = OTFE_Decrypt_Header(headerCipher, headerPlain, 512, (byte *)szPwd, _tcslen(szPwd));
	if (!bSuccess) {
		printf("Not an OTF volume or incorrect password\n");
		CloseHandle(m_hVolume);
		SecureZeroMemory(headerPlain, sizeof(headerPlain));
		if (szPwd)
			SecureZeroMemory(szPwd, strlen(szPwd));
		return 1;
	}

	if (crypt_ctx->bHidden == TRUE)
		crypt_ctx->m_i64Offset = m_i64FileSz.QuadPart - crypt_ctx->m_i64Offset - 1536;

	partition = _FAT_partition_constructor(m_hVolume, 8);

	if (partition == NULL) {
		printf("Does not appear to be a FAT partition\n");
		SecureZeroMemory(headerPlain, sizeof(headerPlain));

		for (int i = 0; i < crypt_ctx->cipher_count; i++) {
			ecb_done(&crypt_ctx->ci_ecb[i]);
		}
		SecureZeroMemory(crypt_ctx, sizeof(*crypt_ctx));
		delete crypt_ctx;

		if (szPwd)
			SecureZeroMemory(szPwd, strlen(szPwd));

		return 1;
	} else if (partition->filesysType == FS_UNKNOWN) {
		printf("Does not appear to be a FAT partition\n");
		SecureZeroMemory(headerPlain, sizeof(headerPlain));
		_FAT_partition_destructor(partition);
		SecureZeroMemory(partition, sizeof(*partition));

		for (int i = 0; i < crypt_ctx->cipher_count; i++) {
			ecb_done(&crypt_ctx->ci_ecb[i]);
		}
		SecureZeroMemory(crypt_ctx, sizeof(*crypt_ctx));
		delete crypt_ctx;

		if (szPwd)
			SecureZeroMemory(szPwd, strlen(szPwd));

		return 1;
	}

	fileBuf = new char[1024*1024];

	//Export File
	if (method == 0) {
		m_hFile = CreateFile(szDest, GENERIC_READ | GENERIC_WRITE,
				FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
				/*CREATE_ALWAYS*/CREATE_NEW, NULL/*FILE_FLAG_NO_BUFFERING*/,
				NULL);

		if (m_hFile == INVALID_HANDLE_VALUE) {
			LPVOID lpMsgBuf;

			FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
						  NULL,
						  GetLastError(),
						  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
						  (LPTSTR) &lpMsgBuf,
						  0,
						  NULL);

			_tprintf("Error creating file\n%s\n%s\n", (LPCTSTR)lpMsgBuf, szDest);

			LocalFree(lpMsgBuf);
		} else {
			flags = 0;
			flags |= O_RDONLY;
			mode = 0;

			memcpy(strSource, szSource, sizeof(strSource));

			fd = _FAT_open_r(&r, &mFile, strSource, flags, mode);
			if (fd == -1) {
				printf("Unable to open file: %d\n%s\n%s\n", r._errno, error_code(r._errno), strSource);
			} else {
				while ((retVal = _FAT_read_r(&r, fd, fileBuf, 1024*1024)) > 0) {
					WriteFile(m_hFile, fileBuf, retVal, &dwBytesWritten, FALSE);
				}

				retVal = _FAT_close_r(&r, fd);
			}
			CloseHandle(m_hFile);
		}
	}

	//Import file
	if (method == 1) {
		m_hFile = CreateFile(szSource, GENERIC_READ,
				FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
				OPEN_EXISTING, NULL/*FILE_FLAG_NO_BUFFERING*/,
				NULL);

		if (m_hFile == INVALID_HANDLE_VALUE) {
			LPVOID lpMsgBuf;

			FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
						  NULL,
						  GetLastError(),
						  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
						  (LPTSTR) &lpMsgBuf,
						  0,
						  NULL);

			_tprintf("Error opening file\n%s\n%s\n", (LPCTSTR)lpMsgBuf, szDest);

			LocalFree(lpMsgBuf);
		} else {
			flags = 0;
			flags |= O_CREAT | O_RDWR;
			mode = 0;

			//wcstombs(strDest, szDest, sizeof(strDest)); 
			memcpy(strDest, szDest, sizeof(strDest));

			fd = _FAT_open_r(&r, &mFile, strDest, flags, mode);
			if (fd == -1) {
				printf("Unable to create file: %d\n%s\n%s\n", r._errno, error_code(r._errno), strDest);
			} else {
				retVal = 1;
				while (retVal > 0) {
					if (ReadFile(m_hFile, fileBuf, 1024*1024, &dwBytesWritten, FALSE) == 0) {
						LPVOID lpMsgBuf;

						FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
									NULL,
									GetLastError(),
									MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
									(LPTSTR) &lpMsgBuf,
									0,
									NULL);

						_tprintf("Error reading file\n%s\n%s\n", (LPCTSTR)lpMsgBuf, szDest);
						LocalFree(lpMsgBuf);
						break;
					}
					retVal = _FAT_write_r (&r, fd, fileBuf, dwBytesWritten);
				}

				retVal = _FAT_close_r(&r, fd);
			}

			CloseHandle(m_hFile);
		}
	}

	//rename/move file/folder
	if (method == 2) {
		mode = 0;
		//wcstombs(strSource, szSource, sizeof(strSource)); 
		//wcstombs(strDest, szDest, sizeof(strDest));
		memcpy(strSource, szSource, sizeof(strSource));
		memcpy(strDest, szDest, sizeof(strDest));

		retVal = _FAT_rename_r(&r, strSource, strDest);
		if (retVal == -1) {
			printf("Unable to rename: %d\n%s\n%s\n%s\n", r._errno, error_code(r._errno), strSource, strDest);
		}
	}

	//create folder
	if (method == 3) {
		mode = 0;
		//wcstombs(strDest, szSource, sizeof(strDest));
		memcpy(strDest, szSource, sizeof(strDest));

		retVal = retVal = _FAT_mkdir_r(&r, strDest, mode);
		if (retVal == -1) {
			printf("Unable to create folder: %d\n%s\n%s\n", r._errno, error_code(r._errno), strDest);
		}
	}

	//Delete file/folder
	if (method == 4) {
		//wcstombs(strSource, szSource, sizeof(strSource)); 
		memcpy(strSource, szSource, sizeof(strSource));

		retVal = _FAT_unlink_r(&r, strSource);
		if (retVal == -1) {
			printf("Unable to delete file: %d\n%s\n%s\n", r._errno, error_code(r._errno), strSource);
		}
	}

	//list all files/folders
	if (method == 5) {
		ListAll_Dirs("/");
	}

	if (method == 6) {
		printf("Partition Type: ");
		if (partition->filesysType == FS_FAT12)
			printf("Fat12\n");
		else if (partition->filesysType == FS_FAT16)
			printf("Fat16\n");
		else if (partition->filesysType == FS_FAT32)
			printf("Fat32\n");
		else
			printf("Unknown\n");

		printf("Fat size: %0.2f MB,\n  %I64u bytes\n",
			(UINT64)partition->fat.sectorsPerFat * partition->bytesPerSector / 1024.0 / 1024.0,
			(UINT64)partition->fat.sectorsPerFat * partition->bytesPerSector);

		printf("Cluster Size: %0.2f KB,\n  %u bytes\n",
			partition->bytesPerCluster / 1024.0, partition->bytesPerCluster);

		printf("Size: %0.2f MB,\n  %I64u bytes\n",
			(UINT64)partition->fat.lastCluster * partition->bytesPerCluster / 1024.0 / 1024.0,
			(UINT64)partition->fat.lastCluster * partition->bytesPerCluster);

		printf("Used: %0.2f MB,\n  %I64u bytes\n",
			(((UINT64)partition->fat.lastCluster * partition->bytesPerCluster)
				- ((UINT64)partition->fsinfo.freeClusters * partition->bytesPerCluster))
				/ 1024.0 / 1024.0,
			((UINT64)partition->fat.lastCluster* partition->bytesPerCluster)
				- ((UINT64)partition->fsinfo.freeClusters * partition->bytesPerCluster));

		printf("Free: %0.2f MB,\n  %I64u bytes\n",
			(UINT64)partition->fsinfo.freeClusters * partition->bytesPerCluster / 1024.0 / 1024.0,
			(UINT64)partition->fsinfo.freeClusters * partition->bytesPerCluster);
	}

	//Clear everything
	CloseHandle(m_hVolume);
	SecureZeroMemory(headerPlain, sizeof(headerPlain));
	SecureZeroMemory(headerCipher, sizeof(headerCipher));

	if (szPwd)
		SecureZeroMemory(szPwd, strlen(szPwd));

	for (int i = 0; i < crypt_ctx->cipher_count; i++) {
		ecb_done(&crypt_ctx->ci_ecb[i]);
	}
	SecureZeroMemory(crypt_ctx, sizeof(*crypt_ctx));
	delete crypt_ctx;

	SecureZeroMemory(fileBuf, sizeof(1024*1024));
	_FAT_partition_destructor(partition);
	SecureZeroMemory(partition, sizeof(*partition));
	
	delete [] fileBuf;

	return 0;
}
