/*
 crypto.cpp
 Cryptographic functions for TC container

 Copyright (c) 2006 Josh Harris
	
 Redistribution and use in source and binary forms, with or without modification,
 are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice,
     this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright notice,
     this list of conditions and the following disclaimer in the documentation and/or
     other materials provided with the distribution.
  3. The name of the author may not be used to endorse or promote products derived
     from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE
 LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

	2006-10-18 - JHH
		* Original release
*/

#include "crypto.h"

#include <math.h>

OTFE_CONTEXT *crypt_ctx;

inline void xor128(byte *out, byte * in)
{
	UINT64 *in64 = (UINT64 *)in;
	UINT64 *out64 = (UINT64 *)out;
	*out64 ^= *in64;

	in64 = (UINT64 *)&in[8];
	out64 = (UINT64 *)&out[8];
	*out64 ^= *in64;
}

inline void xor64(byte *out, byte *in)
{
	UINT64 *in64 = (UINT64 *)in;
	UINT64 *out64 = (UINT64 *)out;
	*out64 ^= *in64;
}

byte FlipBits(byte a)
{
	return	(a & 0x01) << 7 |
			(a & 0x02) << 5 |
			(a & 0x04) << 3 |
			(a & 0x08) << 1 |
			(a & 0x10) >> 1 |
			(a & 0x20) >> 3 |
			(a & 0x40) >> 5 |
			(a & 0x80) >> 7;
}

void FlipBytes(byte *in, byte *out, int size)
{
	for (int i = 0; i < size; i++)
		out[i] = in[size-1-i];
}

void FlipBytesST(byte *in, int size)
{
	byte tmp;
	for (int i = 0; i < size / 2; i++) {
		tmp = in[i];
		in[i] = in[size - i - 1];
		in[size - i - 1] = tmp;
	}
}

void MirrorBytes(byte *in, byte *out, int size)
{
	FlipBytes(in, out, size);
	for (int i = 0; i < size; i++)
		out[i] = FlipBits(out[i]);
}

int OTFE_Decrypt_Header(byte *intext, byte *plaintext, int len, byte *pwd, int pwd_len)
{
	BOOL bSuccess = FALSE;
	byte key_exp[256], key[96], salt[64];
	INT64 m_i64CurSz = 0, m_i64NewSz = 0;
	int m_iBlkSz, m_iKeySz;

	int mode, iter_count, cipher_count;
	ULONG outlen;

	VirtualLock(&key_exp, 256);
	VirtualLock(&key, 96);

	outlen = 152;

	for (int header = 0; header < 2; header++) {
		int header_offset = 0;
		if (header == 1)
			header_offset = 512;

		SecureZeroMemory(salt, sizeof(salt));
		memcpy(salt, &intext[header_offset], 64);

		for (int hash = 0; hash < TC_HASH_COUNT && !bSuccess; hash++) {
			//if (hash != 0)
				//continue;

			crypt_ctx = new OTFE_CONTEXT;
			memset(crypt_ctx, 0, sizeof(*crypt_ctx));
			crypt_ctx->bHeader = TRUE;

			if (hash == 0) {
				crypt_ctx->hash_idx = find_hash("rmd160");
				iter_count = 2000;
			} else if (hash == 1) {
				crypt_ctx->hash_idx = find_hash("sha1");
				iter_count = 2000;
			} else if (hash == 2) {
				crypt_ctx->hash_idx = find_hash("whirlpool");
				iter_count = 1000;
			}

			if ((errno = pkcs_5_alg2(pwd, pwd_len,
					salt, 64,
					iter_count, crypt_ctx->hash_idx,
					key_exp, &outlen)) != CRYPT_OK) {
				printf("Error hashing key: %s\n", error_to_string(errno));
				return 0;
			}

			m_iBlkSz = 16;
			m_iKeySz = 32;

			memcpy(key, &key_exp[DISK_IV_SIZE], 96);
			memcpy(&crypt_ctx->lrwKey, key_exp, m_iBlkSz);

			// for libTomCrypt LRW, mirrot bit order
			MirrorBytes(&crypt_ctx->lrwKey[0], &crypt_ctx->temp[0], 16);
			memcpy(crypt_ctx->lrwKey, crypt_ctx->temp, 16);

			for (mode = 0; mode < TC_MODE_COUNT && !bSuccess; mode++) {
				if (mode != OTF_LRW) //Skip CBC mode for now
					continue;
				for (int ciper_id = 0; ciper_id < TC_CIPHER_COUNT + 5 && !bSuccess; ciper_id++) {
					//if (ciper_id != )
					//	continue;

					int cipher = ciper_id;

					if (cipher >= TC_CIPHER_COUNT)
						cipher = OTF_AES;

					m_iBlkSz = cipher_descriptor[cipher].block_length;
					m_iKeySz = cipher_descriptor[cipher].max_key_length;
					cipher_descriptor[cipher].name;

					if (mode == OTF_LRW) {
						if (ciper_id == OTF_AES_TWOFISH) {
							//AES-Twofish
							crypt_ctx->cipher_count = cipher_count = 2;
							crypt_ctx->cipher_idx[0] = 4;
							crypt_ctx->cipher_idx[1] = 0;
						} else if (ciper_id == OTF_AES_TWOFISH_SERPENT) {
							//AES-Twofish-Serpent
							crypt_ctx->cipher_count = cipher_count = 3;
							crypt_ctx->cipher_idx[0] = 3;
							crypt_ctx->cipher_idx[1] = 4;
							crypt_ctx->cipher_idx[2] = 0;
						} else if (ciper_id == OTF_SERPENT_AES) {
							//Serpent-AES
							crypt_ctx->cipher_count = cipher_count = 2;
							crypt_ctx->cipher_idx[0] = 0;
							crypt_ctx->cipher_idx[1] = 3;
						} else if (ciper_id == OTF_SERPENT_TWOFISH_AES) {
							//Serpent-Twofish-AES
							crypt_ctx->cipher_count = cipher_count = 3;
							crypt_ctx->cipher_idx[0] = 0;
							crypt_ctx->cipher_idx[1] = 4;
							crypt_ctx->cipher_idx[2] = 3;
						} else if (ciper_id == OTF_TWOFISH_SERPENT) {
							//Twofish-Serpent
							crypt_ctx->cipher_count = cipher_count = 2;
							crypt_ctx->cipher_idx[0] = 3;
							crypt_ctx->cipher_idx[1] = 4;
						} else {
							crypt_ctx->cipher_count = cipher_count = 1;
							crypt_ctx->cipher_idx[0] = ciper_id;
						}

						for (int i = 0; i < cipher_count; i++) {
							if ((errno = ecb_start(crypt_ctx->cipher_idx[i], &key[m_iKeySz*i], m_iKeySz, 0, &crypt_ctx->ci_ecb[i])) != CRYPT_OK) {
								printf("ecb_start error: %s\n", error_to_string(errno));
								//exit(-1);
								break;
							}
						}

						OTFE_LRW(crypt_ctx, OTF_DECRYPT, &intext[64+header_offset], plaintext, 512-64, 1);

						for (int i = 0; i < cipher_count; i++) {
							ecb_done(&crypt_ctx->ci_ecb[i]);
						}
					} else if (mode == OTF_CBC) {
						/*if (ciper_id == OTF_BLOWFISH) {
							FlipBytesST(IV1, 4);
							FlipBytesST(&IV1[4], 4);
						} else if (ciper_id >= TC_CIPHER_COUNT) {
							continue;
						}

						if ((errno = cbc_start(k, IV1, key, m_iKeySz, 0, &cbc)) != CRYPT_OK) {
							printf("cbc_start error: %s\n", error_to_string(errno));
							//exit(-1);
							continue;
						}

						TC_Decrypt_CBC(&ciphertext[64], plaintext, 512-64, &IV1[8], 8, &cbc);

						cbc_done(&cbc);*/
					}

					if (plaintext[0] == 'T' && plaintext[1] == 'R'
							&& plaintext[2] == 'U' && plaintext[3] == 'E') {
						bSuccess = TRUE;

						SecureZeroMemory(crypt_ctx->lrwKey, sizeof(crypt_ctx->lrwKey));
						for (int i = 0; i < crypt_ctx->cipher_count; i++) {
							SecureZeroMemory(&crypt_ctx->ci_ecb[i], sizeof(crypt_ctx->ci_ecb[i]));
						}

						crypt_ctx->enc_type = ciper_id;

						memcpy(&crypt_ctx->lrwKey, &plaintext[256-64], m_iBlkSz);

						// for libTomCrypt LRW, mirrot bit order
						MirrorBytes(&crypt_ctx->lrwKey[0], &crypt_ctx->temp[0], 16);
						memcpy(crypt_ctx->lrwKey, crypt_ctx->temp, 16);

						if (header == 0) {
							crypt_ctx->m_i64Offset = 512;
							crypt_ctx->bHidden = FALSE;
						}
						else {
							crypt_ctx->bHidden = TRUE;
							crypt_ctx->m_i64Offset = 512;
							//memcpy(&crypt_ctx->m_i64Offset, &plaintext[92-64], 8);
							FlipBytes(&plaintext[92-64], (byte *)&crypt_ctx->m_i64Offset, 8);
						}

						for (int i = 0; i < crypt_ctx->cipher_count; i++) {
							if ((errno = ecb_start(crypt_ctx->cipher_idx[i], &plaintext[288-64 + m_iKeySz*i], m_iKeySz, 0, &crypt_ctx->ci_ecb[i])) != CRYPT_OK) {
								printf("ecb_start error: %s\n", error_to_string(errno));
								//exit(-1);
								break;
							}
						}
					} //if (plaintext[0] == 'T' && plaintext[1] == 'R'
				} //for (int ciper_id = 0; ciper_id < TC_CIPHER_COUNT
			} //for (j = 0; j < TC_MODE_COUNT && !bSuccess; j++) {
		} //for (i = 0; i < TC_HASH_COUNT && !bSuccess; i++) {
	}

	SecureZeroMemory(key_exp, sizeof(key_exp));
	SecureZeroMemory(key, sizeof(key));

	crypt_ctx->bHeader = FALSE;

	return bSuccess;
}

int OTFE_Block(OTFE_CONTEXT *ctx, int dir, byte *intext, byte *outtext, int len, INT64 idx)
{
	if (ctx->mode_idx == OTF_LRW)
		return OTFE_LRW(ctx, dir, intext, outtext, len, idx);
	else
		return 0;
}

int OTFE_LRW(OTFE_CONTEXT *ctx, int dir, byte *intext, byte *outtext, int len, INT64 idx)
{
	byte intextBlk[16];
	int m_iBlkSz = ctx->ci_ecb[0].blocklen;

	VirtualLock(&intextBlk, 16);

	for (int i = 0; i < len/m_iBlkSz; i++) {
		//memset(ctx->lrwIdx, 0, sizeof(ctx->lrwIdx));
		FlipBytes((byte *)&idx, &ctx->lrwIdx[0], 8);

		memcpy(intextBlk, &intext[i*m_iBlkSz], m_iBlkSz);

		if (m_iBlkSz == 16) {
			MirrorBytes(&ctx->lrwIdx[0], &ctx->temp[0], 8);
			memcpy(ctx->lrwIdx, ctx->temp, 8);


			gcm_gf_mult(ctx->lrwKey, ctx->lrwIdx, ctx->BLK_T);

			MirrorBytes(&ctx->BLK_T[0], &ctx->temp[0], 16);
			memcpy(ctx->BLK_T, ctx->temp, 16);

			xor128(intextBlk, ctx->BLK_T);
		} else {
			MirrorBytes(&ctx->lrwIdx[0], &ctx->temp[0], 8);
			memcpy(ctx->lrwIdx, ctx->temp, 8);

			gcm_gf_mult64(&ctx->lrwKey[8], ctx->lrwIdx, ctx->BLK_T);
			MirrorBytes(&ctx->BLK_T[0], &ctx->temp[0], 8);
			memcpy(ctx->BLK_T, ctx->temp, 8);

			xor64(intextBlk, ctx->BLK_T);
		}

		if (dir == OTF_ENCRYPT) {
			for (int j = 0; j < ctx->cipher_count; j++) {
				OTFE_ECB(dir, intextBlk, &outtext[i*m_iBlkSz], m_iBlkSz, &ctx->ci_ecb[j]);
				memcpy(intextBlk, &outtext[i*m_iBlkSz], m_iBlkSz);
			}
		} else {
			for (int j = ctx->cipher_count - 1; j >= 0 ; j--) {
				OTFE_ECB(dir, intextBlk, &outtext[i*m_iBlkSz], m_iBlkSz, &ctx->ci_ecb[j]);
				memcpy(intextBlk, &outtext[i*m_iBlkSz], m_iBlkSz);
			}
		}


		if (m_iBlkSz == 16) {
			xor128(&outtext[i*m_iBlkSz], ctx->BLK_T);
		} else {
			xor64(&outtext[i*m_iBlkSz], ctx->BLK_T);
		}

		idx++;

		// For Header Only
		if (i == 0 && dir == 1 && ctx->bHeader == TRUE) {
			if (outtext[0] != 'T' && outtext[1] != 'R'
					&& outtext[2] != 'U' && outtext[3] != 'E') {
				break;
			}
		}
	}

	SecureZeroMemory(intextBlk, sizeof(intextBlk));

	return 1;
}

int OTFE_ECB(int dir, byte *intext, byte *outtext, int len, symmetric_ECB *ecb)
{
	if (ecb->cipher == OTF_BLOWFISH) {
		FlipBytesST(intext, 4);
		FlipBytesST(&intext[4], 4);
	}

	if (dir == OTF_ENCRYPT) {
		if ((errno = ecb_encrypt(intext, outtext, len, ecb)) != CRYPT_OK) {
			printf("ecb_encrypt error: %s\n", error_to_string(errno));
			//exit(-1);
			return 0;
		}
	} else {
		if ((errno = ecb_decrypt(intext, outtext, len, ecb)) != CRYPT_OK) {
			printf("ecb_decrypt error: %s\n", error_to_string(errno));
			//exit(-1);
			return 0;
		}
	}

	if (ecb->cipher == OTF_BLOWFISH) {
		FlipBytesST(outtext, 4);
		FlipBytesST(&outtext[4], 4);
	}

	return 1;
}
