/*******************************************************************************
 * Copyright (C) 2005 Chris Miles
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 ******************************************************************************/
package org.bbcorrector;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.microedition.io.Connector;
import javax.microedition.io.HttpConnection;
import javax.microedition.io.HttpsConnection; // Added 2006-06-18, v1.2.2

import net.rim.device.api.util.StringMatch; // Added 2006-03-25, v1.2
import net.rim.device.api.util.Arrays; // Added 2006-03-25, v1.2
//import net.rim.blackberry.api.browser.URLEncodedPostData; // Added 2006-03-25, v1.2, removed 2006-04-02, v1.2

import net.rim.blackberry.api.menuitem.ApplicationMenuItem;
import net.rim.device.api.ui.Field;
import net.rim.device.api.ui.UiApplication;
import net.rim.device.api.ui.component.ActiveAutoTextEditField;
import net.rim.device.api.ui.component.Dialog;
import net.rim.device.api.ui.component.Status;
import net.rim.device.api.xml.parsers.DocumentBuilder;
import net.rim.device.api.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

public class CheckSpellingMenuItem extends ApplicationMenuItem {

    private BBCorrectorOptions options = null; // Movied here 2006-03-25, v1.2
    private int mAdjustErrorCount = 0; // Added 2006-03-25, v1.2

    CheckSpellingMenuItem() {
        super(5);
    }

    public String toString() {
        return "Spelling";
    }

    public Object run(Object context) {

        new Thread(new Runnable() {

            public void run() {
                // Determine what field has focus
                Field field = UiApplication.getUiApplication().getActiveScreen().getFieldWithFocus();
                // Check that this field is valid for Spell Checking
                if (field instanceof ActiveAutoTextEditField) {
                    options = BBCorrectorOptions.load(); // Moved here 2006-03-25, v1.2

                    ActiveAutoTextEditField editField = (ActiveAutoTextEditField) field;
                    String body = editField.getText();

                    final SpellResult.SpellData spellData = checkSpelling(body, editField);

                    // Added 2006-03-25, v1.2
                    // if auto advance, run the auto advance dialog
                    if (options.bAutoAdvance && spellData != null) {
                        UiApplication.getUiApplication().invokeLater(new Runnable() {
                            public void run() {
                                doSpellingDialog(spellData);
                            }
                        });
                    }
                } else
                    UiApplication.getUiApplication().invokeLater(new Runnable() {

                        public void run() {
                            Status.show("Cannot check spelling on this field...");
                        }
                    });
            }
        }).start();

        return null;
    }

    // Added 2006-03-25, v1.2
    private void doSpellingDialog(SpellResult.SpellData spellData) {
        if (spellData.spellErrors.length > 0) {
            BBCorrectorSpellingDialog dialog = new BBCorrectorSpellingDialog();
            dialog.show(spellData, 0);
            if (dialog.isCancelled()) {
                return;
            }
        }
    }

    private SpellResult.SpellData checkSpelling(final String text,
                               final ActiveAutoTextEditField field) {

        final SpellResult.SpellDataHolder spellDataHolder = new SpellResult.SpellDataHolder();

        BusyDialogWorkerThread workerThread = new BusyDialogWorkerThread() {

            protected void execute() {
                try {
                    //System.out.println("Checking text:\n" + text);
                    String xmlPacket = getDataFromAspell(text);
                    Document document = parseXMLPacket(xmlPacket);

                    SpellResult.SpellData spellData = getSpellResultData(document, text);
                    spellData.textLen = text.length();
                    spellDataHolder.spellData = spellData;
                } catch (Exception e) {
                    spellDataHolder.spellData = null;
                    final Exception theException = e;
                    UiApplication.getUiApplication().invokeLater(new Runnable() {

                        public void run() {
                            Dialog.alert("BBCorrector Error: "
                                    + theException.getMessage());
                        }
                    });
                }
            }
        };
        BusyDialog dialog = new BusyDialog("Checking Spelling...", workerThread);
        dialog.display();

        // Associate spelling result data to this field, if we
        // have misspellings
        SpellResult.SpellData spellData = spellDataHolder.spellData;
        if (spellData != null) {
            // ReAdjust the spellErrors array count, if necessary
            // This would happen if words were skipped over because
            // they are in the custom Dictionary
            SpellResult.SpellData.SpellError nullError = null;
            /*int mErrorCount = spellData.spellErrors.length;
            for (int i = 0; i < mErrorCount - mAdjustErrorCount; i++)
                Arrays.remove(spellData.spellErrors, nullError);*/
            while (Arrays.contains(spellData.spellErrors, nullError))
                Arrays.remove(spellData.spellErrors, nullError);
            //mErrorCount = spellData.spellErrors.length;

            if (spellData.spellErrors.length > 0) {
                final int firstPos = spellData.spellErrors[0].position; // Changed 2006-03-25, v1.2
                SpellResult.getInstance().fieldMap.put(new Integer(field.hashCode()),
                                                       spellData);
                UiApplication.getUiApplication().invokeAndWait(new Runnable() {

                    public void run() {
                        // The following lines and ordering are
                        // important. Due to the retrieval of
                        // the spelling data being performed in
                        // the background, the user can move the focus
                        // away from this field. Before we do a scan we
                        // must have focus on this field for the pattern
                        // matching to work. The act of removeFocus()
                        // then setFocus() causes the field to highlight
                        // the regions
                        field.getScreen().removeFocus();
                        field.setFocus();
                        // We do run() rather than
                        // scanForActiveRegions() because run() is a
                        // foreground process
                        field.run();

                        // Move cursor position to first misspelled word
                        field.setCursorPosition(firstPos); // Added 2006-03-25, v1.2
                    }
                });
            } else {
                UiApplication.getUiApplication().invokeLater(new Runnable() {

                    public void run() {
                        Status.show("No spelling errors found...");
                    }
                });
            }
        }

        return spellData;
    }

    private String getDataFromAspell(final String text) throws Exception {

        final int MAX_RETRY = 5;

        final StringBuffer sb = new StringBuffer();

        // Removed 2006-06-18, v1.2.2
        /*
        HttpConnection c = null;
        InputStream is = null;
        OutputStream os = null;
        */

        String data;

        //BBCorrectorOptions options = BBCorrectorOptions.load(); // Moved 2006-03-25, v1.2
        String url = options.urlCorrectorServer;

        // Added 2006-03-25, v1.2
        String language = ((options.language != null)
            && (options.language.trim().length() > 0) ? options.language
            : "en");

        // Added 2006-03-25, v1.2
        // Replace a few characters that cause problems with the spell check
        // This should not have any affect on the spelling
        String fixedText = text.replace('&', '-');
        fixedText = fixedText.replace('\\', '-');
        fixedText = fixedText.replace('#', '-');
        fixedText = fixedText.replace('<', '-');

        //URLEncodedPostData mURLData = new URLEncodedPostData(null, false);

        // Added 2006-03-25, v1.2
        if (options.useGoogle) {
            //Replace & with space
            data = "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?><spellrequest textalreadyclipped=\"0\" ignoredups=\"0\" ignoredigits=\"1\" ignoreallcaps=\"0\"><text>"
                + fixedText + "</text></spellrequest>";
            String mPage = "/tbproxy/spell?lang=" + language + "&hl=" + language;
            url = "https://www.google.com" + mPage; // Changed 2006-06-18, v1.2.2
        } else {
            /*mURLData.append("lang", language); //"This is the \\!@#$%^&*()_+"
            mURLData.append("check", fixedText);
            data = mURLData.toString();*/

            data = "lang=" + language + "&" + "check=" + HttpUtils.encodeURL(fixedText);
        }

        if (options.disableMDSProxy)
            url += ";deviceside=true";
        else
            url += ";deviceside=false";
        String username = ((options.username != null)
                && (options.username.trim().length() > 0) ? options.username
                : null);
        String password = ((options.password != null)
                && (options.password.trim().length() > 0) ? options.password
                : null);

        int responseCode = getUrlData(url, data, sb, username, password);
        int retry = 0;
        while ((responseCode == HttpConnection.HTTP_UNAUTHORIZED)
                && (retry++ < MAX_RETRY)) {
            // We got a unauthorized resonse, so try again sending our
            // credentials
            responseCode = getUrlData(url, data, sb, username, password);
        }
        // If we still have a response code of HTTP_UNAUTHORIZED at this point
        // then trow an exception to let the user know. Most likely an incorrect
        // username/password has been entered
        if (responseCode == HttpConnection.HTTP_UNAUTHORIZED)
            throw new IOException("HTTP response code " + responseCode
                    + " received. Please check your Username and Password");

        String xmlPacket = sb.toString();
        //System.out.println("*** url data=\n" + xmlPacket);

        return xmlPacket;
    }

    private int getUrlData(String url,
                           String data,
                           StringBuffer sb,
                           String username,
                           String password) throws IOException {

        HttpConnection c = null;
        // Added 2006-06-18, v1.2.2
        HttpsConnection cs = null;
        InputStream is = null;
        OutputStream os = null;

        try {
            // Added 2006-03-25, v1.2
            // Changed to https 2006-06-18, v1.2.2
            if (options.useGoogle) {
                String mPage = "/tbproxy/spell?lang=en&hl=en";
                //System.out.println("*** BBCorrector Server url=" + url);
                //System.out.println(data);
                cs = (HttpsConnection) Connector.open(url);
                cs.setRequestMethod(HttpConnection.POST);
                if ((username != null) && (password != null)) {
                    //System.out.println("*** Setting up HTTP Basic Authentication user="
                    //        + username + " pwd=" + password);
                    cs.setRequestProperty("Authorization", "Basic "
                            + HttpUtils.base64Encode(username + ":" + password));
                }
                cs.setRequestProperty("POST", mPage + " HTTP/1.0");
                cs.setRequestProperty("MIME-Version", "1.0");
                //cs.setRequestProperty("Content-type", "application/PTI26");
                cs.setRequestProperty("Content-Type", "text/xml; charset=ISO-8859-1"); //ISO-8859-1s
                //cs.setRequestProperty("Content-type", "application/x-www-form-urlencoded; charset=\"UTF-8\"");
                //cs.setRequestProperty("Content-type", "application/octet-stream");
                cs.setRequestProperty("Content-length", "" + data.length());
                cs.setRequestProperty("Request-number", "1");
                cs.setRequestProperty("Document-type", "Request");
                cs.setRequestProperty("Interface-Version", "Test 1.4");
                cs.setRequestProperty("Connection", "close");
                //cs.setRequestProperty("Content-Type",
                //                    "application/x-www-form-urlencoded");
                os = cs.openOutputStream();
                os.write(data.getBytes());
                os.flush();
            } else {
                //System.out.println("*** BBCorrector Server url=" + url);
                c = (HttpConnection) Connector.open(url);
                c.setRequestMethod(HttpConnection.POST);
                if ((username != null) && (password != null)) {
                    //System.out.println("*** Setting up HTTP Basic Authentication user="
                    //        + username + " pwd=" + password);
                    c.setRequestProperty("Authorization", "Basic "
                            + HttpUtils.base64Encode(username + ":" + password));
                }
                c.setRequestProperty("Content-Type",
                                    "application/x-www-form-urlencoded");
                c.setRequestProperty("Content-Length", "" + data.length());
                os = c.openOutputStream();
                os.write(data.getBytes());
                os.flush();
            }

            //System.out.println("Encoded text:\n" + data);

            // Check response code
            int responseCode;
            if (options.useGoogle) {
                responseCode = cs.getResponseCode();
            } else {
                responseCode = c.getResponseCode();
            }
            //System.out.println("*** Http response code=" + responseCode);
            switch (responseCode) {
            case HttpConnection.HTTP_OK:
                // Changed 2006-06-18, v1.2.2
                if (options.useGoogle) {
                    is = cs.openInputStream();
                } else {
                    is = c.openInputStream();
                }

                int ch;
                while ((ch = is.read()) != -1) {
                    sb.append((char) ch);
                }

                break;
            case HttpConnection.HTTP_UNAUTHORIZED:
                // We only support Basic Authentication (we assume this)
                System.out.println("***** Authenticate header="
                        + c.getHeaderField("WWW-Authenticate"));
                break;
            default:
                throw new IOException("HTTP response code: " + responseCode);
            }

            return responseCode;

        } finally {
            if (os != null) {
                try {
                    os.close();
                } catch (Exception e) {
                }
            }
            if (is != null) {
                try {
                    is.close();
                } catch (Exception e) {
                }
            }
            if (c != null) {
                try {
                    c.close();
                } catch (Exception e) {
                }
            }
            // Added 2006-06-18, v1.2.2
            if (cs != null) {
                try {
                    cs.close();
                } catch (Exception e) {
                }
            }
        }
    }

    private Document parseXMLPacket(String xmlPacket) throws Exception {

        Document document = null;

        try {
            // Parse XML packet
            // // Added 2006-03-23, v1.2, added "UTF-8"
            // to fix problem with malformed UTF-8 packet errors
            ByteArrayInputStream bis = new ByteArrayInputStream(xmlPacket.getBytes("UTF-8"));
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setIgnoringElementContentWhitespace(true);
            factory.setAllowUndefinedNamespaces(true);
            DocumentBuilder builder = factory.newDocumentBuilder();
            document = builder.parse(bis);
            bis.close();
        } catch (Exception e) {
            throw new Exception("Error parsing Spelling Data ("
                    + e.getMessage() + ")\n" + xmlPacket);
        }

        return document;
    }

    private SpellResult.SpellData getSpellResultData(Document document, String text)
            throws Exception {

        //System.out.println("*** Processing Spelling Results");

        Element root = document.getDocumentElement();

        // Added 2006-03-25, v1.2
        mAdjustErrorCount = 0;
        if (options.useGoogle) {
            // Check for error from Google Server
            final String strError;
            strError = root.getAttribute("error");
            if (strError.equals("1")) {
                // We got an exception from the server
                //System.out.println("*** Unkown Exception from Google server: " + strError);
                throw new Exception("*** Unkown Exception from Google server: " + strError);
            }
        } else {
            // Check for exception from BBCorrector Server
            NodeList exception = root.getElementsByTagName("exception");
            if (exception.getLength() > 0) {
                // We got an exception from the server
                final String msg = exception.item(0).getFirstChild().getNodeValue();
                //System.out.println("*** Exception from server: " + msg);
                throw new Exception(msg);
            }
        }

        NodeList errors = root.getChildNodes();
        SpellResult.SpellData spellData = new SpellResult.SpellData();
        int errorNum = errors.getLength();
        spellData.spellErrors = new SpellResult.SpellData.SpellError[errorNum];
        //System.out.println("*** Number errors=" + errorNum);
        for (int i = 0; i < errorNum; i++) {
            Element error = (Element) errors.item(i);
            SpellResult.SpellData.SpellError spellError = new SpellResult.SpellData.SpellError();

            // Added 2006-03-25, v1.2
            if (options.useGoogle) {
                // Get column number of mispelled word
                spellError.position = Integer.parseInt(error.getAttribute("o"));

                // Get mispelled word
                int mCharStart = spellError.position;
                int mCharEnd = Integer.parseInt(error.getAttribute("l"));
                spellError.wordError = text.substring(mCharStart, mCharStart + mCharEnd);

                // Check for this word in the dictionary and skip over it, if it is there
                //if (options.mvDictionary == null)
                //    options.mvDictionary = new String[0];
                boolean bSkipword = false;
                for (int k = 0; k < options.mvDictionary.length; k++) {
                    if (((String)options.mvDictionary[k]).equalsIgnoreCase(spellError.wordError)) {
                        bSkipword = true;
                        break;
                    }
                }
                if (bSkipword)
                    continue;

                // Get suggestions
                if (error.getFirstChild() != null) {
                    String strSuggestions;
                    strSuggestions = error.getFirstChild().getNodeValue();
                    
                    // Added 2006-04-08 v1.21
                    int mPos = -1;
                    int mPosStart = 0;
                    StringMatch mMatch = new StringMatch("\t");
                    spellError.suggestions = new String[0];

                    mPos = mMatch.indexOf(strSuggestions, mPos + 1);
                    while ( mPos > 0 ) {
                        Arrays.add(spellError.suggestions, strSuggestions.substring(mPosStart, mPos));
                        mPosStart = mPos + 1;
                        mPos = mMatch.indexOf(strSuggestions, mPos + 1);
                    }
                    
                    // Get single item or last element from comma separated list
                    if (strSuggestions.length() > 0)
                        Arrays.add(spellError.suggestions, strSuggestions.substring(mPosStart, strSuggestions.length()));

                    // Removed 2006-04-08 v1.21
                    /*int suggestNum = 0;
                    int mPos = -1;
                    StringMatch mMatch = new StringMatch("\t");
                    //mPos = mMatch.indexOf(strSuggestions, mPos);
                    while ( (mPos = mMatch.indexOf(strSuggestions, mPos + 1)) > 0 ) {
                        suggestNum++;
                    }

                    spellError.suggestions = new String[suggestNum];
                    int mPosStart = 0;
                    mPos = mMatch.indexOf(strSuggestions, 0);
                    for (int j = 0; j < suggestNum; j++) {
                        spellError.suggestions[j] = strSuggestions.substring(mPosStart, mPos);
                        mPosStart = mPos + 1;
                        mPos = mMatch.indexOf(strSuggestions, mPos + 1);
                    }*/
                }
            } else {
                // Get mispelled word
                NodeList nl = error.getElementsByTagName("word");
                if (nl != null)
                    spellError.wordError = nl.item(0).getFirstChild().getNodeValue();

                // Added 2006-03-25, v1.2
                // Check for this word in the dictionary and skip over it, if it is there
                //if (options.mvDictionary == null)
                //    options.mvDictionary = new String[0];
                boolean bSkipword = false;
                for (int k = 0; k < options.mvDictionary.length; k++) {
                    if (((String)options.mvDictionary[k]).equalsIgnoreCase(spellError.wordError)) {
                        bSkipword = true;
                        break;
                    }
                }
                if (bSkipword)
                    continue;

                // Get column number of mispelled word
                nl = error.getElementsByTagName("position");
                if (nl != null)
                    spellError.position = Integer.parseInt(nl.item(0).getFirstChild().getNodeValue());

                // Get suggestions
                nl = error.getElementsByTagName("suggest");
                if (nl != null) {
                    int suggestNum = nl.getLength();
                    spellError.suggestions = new String[suggestNum];
                    for (int j = 0; j < suggestNum; j++) {
                        Element suggest = (Element) nl.item(j);
                        spellError.suggestions[j] = suggest.getFirstChild().getNodeValue();
                    }
                }
            }
            spellData.spellErrors[mAdjustErrorCount] = spellError;

            // Added 2006-03-25, v1.2
            // Because we may have skipped some words that are contained in the custom dictionairy
            mAdjustErrorCount++;
        }

        return spellData;
    }

}
