//	Quicktime Import Plugin for VirtualDub
//	Copyright (C) 2007 Josh Harris (tateu)
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "QTMovieDec.h"

bool SortByDisplayTime(const mpegFrameInfo& S1, const mpegFrameInfo& S2)
{
	return S1.timePlay < S2.timePlay;
}

bool SortByDecodeTime(const mpegFrameInfo& S1, const mpegFrameInfo& S2)
{
	return S1.timeDecode < S2.timeDecode;
}

frameDataStruct				frameData;
myFrameBuffer				m_sFrameBuffer;
vector<myFrameBuffer>		m_vFrameBuffer;

CQTMovieDec::CQTMovieDec()
{
	OSErr err = noErr;
long m_lQTVersion;
bool bQTVersion7;

	m_GWorld = NULL;
	GWorldDataPtr = NULL;
	m_sFrameBuffer.data = NULL;
	pDataCompressed = NULL;
	m_compressedData = NULL;
	m_imageDesc = NULL;
	m_PixMap = NULL;
	m_Movie = NULL;
	m_vTrack = NULL;
	m_vMedia = NULL;
	m_aTrack = NULL;
	m_aMedia = NULL;
	//m_FileName = NULL;
	//m_fileSpec = NULL;
	m_resRefNum = 0;
	//m_vTrackFrame.bottom = 0;
	converter = NULL;
	m_seqID = NULL;
	hic = NULL;
	decompressionSession = NULL;
	source_sound_description_extension = NULL;
	ioData = NULL;
	sampleTable = NULL;
	m_AudioExtractor = NULL;

	bContainsDisplayOffsets = false;
	bUseSampleTable = false;
	bIFrameOnly = true;
	nextIFrame = 1;

	bIsLoaded = false;
	bQTVersion7 = false;

	bVFW = false;

	strStatus = new char[512];
	memset(strStatus, 0, 512);

	sprintf(strStatus, "No Error");

	//kInitializeQTMLDisableDirectSound kInitializeQTMLNoSoundFlag
	if(InitializeQTML(kInitializeQTMLDisableDirectSound) != noErr)
	{
		TerminateQTML();
		sprintf(strStatus, "Unable to Initialize Quicktime: InitializeQTML");
		return;
	}

	if (EnterMovies() != noErr) {
		TerminateQTML();
		sprintf(strStatus, "Unable to Initialize Quicktime: EnterMovies");
		return;
	}

	bIsLoaded = true;

	// Get installed quicktime version
	err = Gestalt(gestaltQuickTime, &m_lQTVersion);
	if ((err == noErr) && (m_lQTVersion >= 0x07000000)) {//0x05020000 = 5.0.2
		bQTVersion7 = true;
	}
}

CQTMovieDec::~CQTMovieDec()
{
	if (converter) {
		SoundConverterClose(converter);
		converter = NULL;
	}

	if (m_seqID) {
		CDSequenceEnd(m_seqID);
		m_seqID = NULL;
	}

	if (m_iMode == 0) {
		//MoviesTask(m_Movie, 0);
		//StopMovie(m_Movie);
	}


	if (hic) {
		ICDecompressEnd(hic);
		ICClose(hic);
		free(pbiSrc);
		hic = NULL;
	}

	if (decompressionSession) {
		ICMDecompressionSessionFlush(decompressionSession);
		ICMDecompressionSessionRelease(decompressionSession);
		decompressionSession = NULL;
	}

	QTFreeMemory();
	ExitMovies();
	TerminateQTML();
}

void CQTMovieDec::QTFreeMemory()
{
	if (strStatus) {
		delete [] strStatus;
		strStatus = NULL;
	}

	QT_FREE_BUFFER(GWorldDataPtr);
	QT_FREE_BUFFER(m_sFrameBuffer.data);
	QT_FREE_BUFFER(pDataCompressed);

	QT_FREE_HANDLE(m_imageDesc);
	if (m_compressedData)
		HUnlock(m_compressedData);
	QT_FREE_HANDLE(m_compressedData);
	//QT_FREE_HANDLE(inputAudio);
	//QT_FREE_HANDLE(outputAudio);
	QT_FREE_HANDLE(source_sound_description_extension);

	for (int i = 0; i < m_vFrameBuffer.size(); i++)
		QT_DELETE_BUFFER(m_vFrameBuffer[i].data);

	m_vFrameBuffer.clear();
	m_vKeyframes.clear();
	m_vFrameInfo.clear();
	m_vFrameInfoDisplay.clear();
	m_vFrameInfoDecode.clear();

	if (m_resRefNum) {
		CloseMovieFile(m_resRefNum);
		m_resRefNum = 0;
	}
	if (m_Movie) {
		DisposeMovie(m_Movie);
		m_Movie = NULL;
	}
	if (m_GWorld) {
		DisposeGWorld(m_GWorld);
		m_GWorld = NULL;
	}

	if (m_AudioExtractor) {
		MovieAudioExtractionEnd(m_AudioExtractor);
		m_AudioExtractor = NULL;

		if (ioData) {
			for (int i = 0; i < ioData->mNumberBuffers; i++) {
				QT_FREE_BUFFER(ioData->mBuffers[i].mData);
				//QT_FREE_BUFFER(chan[i]);
			}
			free(ioData);
			ioData = NULL;
		}
	}

	if (sampleTable) {
		QTSampleTableRelease(sampleTable);
		sampleTable = NULL;
	}
}

char *CQTMovieDec::GetErrorMessage()
{
	return strStatus;
}

HRESULT CQTMovieDec::OpenMovie(char *strFileName, QTCodecInfo mCodec, int vTrackIdx, int aTrackIdx)
{
	OSErr err = noErr;
	Track m_Track;
	Media m_Media;
	OSType mediaType;

	int i;

	m_CodecInfo = mCodec;

	c2pstrcpy(m_FileName, strFileName);
	err = FSMakeFSSpec(0, 0, m_FileName, &m_fileSpec);
	if (err != noErr && err != fnfErr) {
		sprintf(strStatus, "Error with FSMakeFSSpec, err=%d :\n %s", err, strFileName);
		//QTFreeMemory();
		return E_FAIL;
	}

	// Open reference to the movie
	err = OpenMovieFile(&m_fileSpec, &m_resRefNum, fsRdPerm); // fsCurPerm fsRdPerm (read-only), fsWrPerm (write-only), or fsRdWrPerm  (read-write).
	if (err != noErr) {
		sprintf(strStatus, "Unable to Open Movie, err=%d :\n %s", err, m_fileSpec.name);
		//QTFreeMemory();
		return E_FAIL;
	}

	// Create movie object
	err = NewMovieFromFile(&m_Movie, m_resRefNum, nil, nil, newMovieActive, nil);
	if (err != noErr) {
		sprintf(strStatus, "NewMovieFromFile Error:\n %s", m_fileSpec.name);
		//QTFreeMemory();
		return E_FAIL;
	}

	// Set playback flags
	//unsigned flags = 0;
	//flags |= hintsOffscreen | hintsAllowBlacklining | hintsDontDraw | hintsAllowInterlace | hintsDontUseVideoOverlaySurface;
	//flags |= hintsPlayingEveryFrame;
	//SetMoviePlayHints( m_Movie, flags, -1 );

	//SetMoviePlayHints(m_Movie, hintsScrubMode, hintsScrubMode);
	SetMoviePlayHints(m_Movie, hintsAllowInterlace, hintsAllowInterlace);
	//SetMoviePlayHints(m_Movie, hintsAllowBlacklining, hintsAllowBlacklining);
	if (m_CodecInfo.quality > 0)
		SetMoviePlayHints(m_Movie, hintsHighQuality, hintsHighQuality);
	else
		SetMoviePlayHints(m_Movie, 0, hintsHighQuality);

	// Set initial values
	m_vTrackCount = m_aTrackCount = 0;
	m_vMediaFrameCount = m_vMovieFrameCount = 0;
	m_vTrackDur = m_vMediaDur = 0;

	m_vMovieDur = GetMovieDuration(m_Movie);
	m_MovieTimeScale = GetMovieTimeScale(m_Movie);
	m_iTrackCount = GetMovieTrackCount(m_Movie);

	// Loop through all movie tracks looking for the track index
	// that matches what the user selected
	for (i = 1; i <= m_iTrackCount; i++) {
		m_Track = GetMovieIndTrack(m_Movie, i);
		m_Media = GetTrackMedia(m_Track);

		GetMediaHandlerDescription(m_Media, &mediaType, nil, nil);
		if (mediaType == VideoMediaType) {
			m_vTrackCount++;
			if (m_vTrackCount == vTrackIdx) {
				// Found the video track we are looking for
				if (m_CodecInfo.quality > 0)
					SetMediaPlayHints(m_Media, hintsHighQuality, hintsHighQuality);
				else
					SetMediaPlayHints(m_Media, 0, hintsHighQuality);

				// Set global values to current track values
				m_vTrack = m_Track;
				m_vMedia = m_Media;

				// Put video display size into global variable
				m_vTrackFrame.left = 0;
				m_vTrackFrame.top = 0;
				GetMovieBox(m_Movie, &m_vTrackFrame);

				// Get media and track times
				m_vMediaFrameCount = GetMediaSampleCount(m_vMedia);
				m_vTrackDur = GetTrackDuration(m_vTrack);
				m_vMediaDur = GetMediaDuration(m_Media);
				m_vMediaTimeScale = GetMediaTimeScale(m_vMedia);

				m_vMediaSampleDuration = (float)GetMediaDecodeDuration(m_vMedia) / (float)m_vMediaFrameCount;
				m_SampleDuration = (float)GetMediaDecodeDuration(m_vMedia) / (float)m_vMediaFrameCount;
				//m_vMediaSampleDuration = (double)m_vMediaTimeScale / (double)fps;

				TimeValue endTime;
				m_vTrackStartTime = GetTrackOffset(m_vTrack);
				endTime = GetTrackDuration(m_vTrack);
				//m_vTrackIncrement = (endTime - m_vTrackStartTime) / m_vMediaFrameCount; //GetMediaTimeScale(m_vMedia);
				m_vTrackIncrement = (float)m_MovieTimeScale * m_vMediaSampleDuration / (float)m_vMediaTimeScale;
				//m_vTrackIncrement = (float)endTime / m_vMediaFrameCount;

				m_imageDesc=(ImageDescriptionHandle)NewHandleClear(sizeof(ImageDescription));
				GetMediaSampleDescription(m_vMedia, 1, (SampleDescriptionHandle)m_imageDesc);

				m_vTrackFrame.right = (**m_imageDesc).width;
				m_vTrackFrame.bottom = (**m_imageDesc).height;

				// Reset the Movie's display box to the same dimensions as the m_imageDesc
				// Sometimes the display size does not match the actual pixel dimensions stored in m_imageDesc
				SetMovieBox(m_Movie, &m_vTrackFrame);

				m_iDepth = (**m_imageDesc).depth;
				m_CodecInfo.m_iFourCC = (**m_imageDesc).cType;

				if (m_CodecInfo.m_iFourCC == FOUR_CHAR_CODE('AVdv') ||
						m_CodecInfo.m_iFourCC == FOUR_CHAR_CODE('dvc ') ||
						m_CodecInfo.m_iFourCC == FOUR_CHAR_CODE('dvcp'))
					bIsDV25 = true;

				p2cstrcpy(m_CodecInfo.CodecName, (**m_imageDesc).name);
			}
		} else if (mediaType == SoundMediaType) {
			m_aTrackCount++;
			if (m_aTrackCount == aTrackIdx) {
				// Found the audio track we are looking for
				// Set global values to current track values
				m_aTrack = m_Track;
				m_aMedia = m_Media;

				// Get media and track times
				m_aMediaSampleCount = GetMediaSampleCount(m_aMedia);
				m_aMediaDur = GetMediaDuration(m_aMedia);
				m_aTrackDur = GetTrackDuration(m_aTrack);
				//m_aMediaSampleDuration =
				//	(float)GetMediaDecodeDuration(m_aMedia) / (float)m_aMediaDur;

				m_aMediaTimeScale = GetMediaTimeScale(m_aMedia);
				m_AudioTimeBase = GetMovieTimeBase(m_Movie);

				/* create a description of the source sound so we can convert it later */
				//source_sound_description = (SoundDescriptionV1Handle)NewHandle(0);
				source_sound_description = (SoundDescriptionV1Handle)NewHandleClear(sizeof(SoundDescriptionV1Handle));
				_ASSERT(source_sound_description != NULL); /* out of memory */

				GetMediaSampleDescription(m_aMedia, 1,
					(SampleDescriptionHandle)source_sound_description);

				//source_sound_description_extension = NewHandle(0);
				source_sound_description_extension = NewHandleClear(0);
				//assert (source_sound_description_extension != NULL); /* out of memory */

				err = GetSoundDescriptionExtension((SoundDescriptionHandle) source_sound_description,
					&source_sound_description_extension, siDecompressionParams);

				if (err == noErr) {
					/* copy extension to atom format description if we have an extension */
					source_sound_description_extension_size =
						GetHandleSize (source_sound_description_extension);
						HLock (source_sound_description_extension);

					source_sound_decomp_atom = (AudioFormatAtom*)
						NewPtr (source_sound_description_extension_size);
					//assert (source_sound_decomp_atom != NULL); /* out of memory */

					BlockMoveData (*source_sound_description_extension,
						source_sound_decomp_atom,
						source_sound_description_extension_size);

					HUnlock(source_sound_description_extension);
				} else {
					source_sound_decomp_atom = NULL;
				}

				atom = source_sound_decomp_atom;
				source_format.flags = 0;
				source_format.format = (*source_sound_description)->desc.dataFormat;
				source_format.numChannels = (*source_sound_description)->desc.numChannels;
				source_format.sampleSize = (*source_sound_description)->desc.sampleSize;
				//source_format.sampleRate = UInt16((*source_sound_description)->desc.sampleRate >> 16);
				source_format.sampleRate = (*source_sound_description)->desc.sampleRate;
				source_format.sampleCount = GetMediaSampleCount(m_aMedia);
				source_format.buffer = NULL;
				source_format.reserved = 0;

				dest_format.flags = 0;
				//kSoundNotCompressed k16BitLittleEndianFormat
				dest_format.format = kSoundNotCompressed;
				dest_format.numChannels = (*source_sound_description)->desc.numChannels;
				dest_format.sampleSize = 16;//(*source_sound_description)->desc.sampleSize;
				dest_format.sampleRate = (*source_sound_description)->desc.sampleRate; //rate44khz
				dest_format.sampleCount = source_format.sampleCount;
				dest_format.buffer = NULL;
				dest_format.reserved = 0;

				DisposeHandle (source_sound_description_extension);
				DisposeHandle ((Handle)source_sound_description);
			}
		}
	}

	if (m_vMedia == NULL && m_aMedia == NULL) {
		sprintf(strStatus, "\nError with OpenMovie:\n"
			"Could not find video for track %d\n"
			"And audio for track %d\n", vTrackIdx, aTrackIdx);
		return E_FAIL;
	}

	return S_OK;
}

bool CQTMovieDec::SetFormat(OSType mFormat, int mRaw, int mMode, float mGamma, int mOrder)
{
	//if ( mFormat == k24BGRAPixelFormat || mFormat == k32BGRAPixelFormat) {
		m_pixelFormat = mFormat;
		m_iRowOrder = mOrder;
		m_iRaw = mRaw;
		m_iMode = mMode;
		m_fGamma = mGamma;
		return true;
	//}

//This does not work because not all Codecs seem to have a cpix resource. Avid Codecs
/*
	bool bCanDecode = false;
	ComponentDescription cd = { decompressorComponentType, 0, 0, 0, cmpIsMissing };
	Component decompressor = 0;

	cd.componentSubType = m_CodecInfo.m_iFourCC;
	decompressor = FindNextComponent(0, &cd);
	if (decompressor) {
		do {
			Handle cpix = NULL;
			if (noErr == GetComponentPublicResource(decompressor,
				FOUR_CHAR_CODE('cpix'), 1, &cpix))
			{
				int i;
				int cpixFormatCount = GetHandleSize(cpix) / sizeof(OSType);
				for (i = 0; i < cpixFormatCount; i++) {
					if (mFormat == (*(OSType**)cpix)[i]) {
						bCanDecode = true;
						m_pixelFormat = mFormat;
						rowOrder = mOrder;
						break;
					}
				}
				DisposeHandle( cpix );
			}
			decompressor = FindNextComponent(decompressor, &cd);
		} while (decompressor && bCanDecode == false);
	}

	return bCanDecode;
*/
}

HRESULT CQTMovieDec::InitializeGraphics(int mAudio, char *strFourCC, bool Yuv2Rgb, int dither)
{
	OSErr err = noErr;
	m_iDither = dither;

	TimeValue64 offset1;
	SInt64 flags = 0;
	mpegFrameInfo myMpegFrameInfo;
	long myFrameCount = 0;
	int i;

	if (m_vTrackCount == 0)
		goto SETUP_AUDIO;

	bContainsDisplayOffsets = MediaContainsDisplayOffsets(m_vMedia);
	sampleTable = NULL;

	// Get the media sample table, use it for IBP codecs
	CopyMediaMutableSampleTable (
		m_vMedia,
		0,
		NULL,
		0,
		0,
		(QTMutableSampleTableRef *)&sampleTable);

	// Assume codec is IFrame only
	// It will be changed if it finds a frame that is not a keyframe in the sample table
	bIFrameOnly = true;

	// Only use the sample table for IBP codecs and not Mode 0 (QTMOVIE_MODE_MTASK)
	//		because if it is used on a ref movie, it will return all video frames
	//		from the source, not just the ones selected in the ref movie
	// This means that IBP ref movies currently do not work
	// Need to figure out how to determine what files are ref movies
	// Also, ref movies, currently return the correct audio samples through VDub,
	//		but the incorrect audio length, they return the full uncut audio length
	// Causes problems in other programs if seeking to time beyond what the ref movie contains
	bUseSampleTable = false;
	/*if (sampleTable && m_iMode != QTMOVIE_MODE_MTASK
			&& ( m_CodecInfo.m_iFourCC == FOUR_CHAR_CODE('avc1')
						|| m_CodecInfo.m_iFourCC == FOUR_CHAR_CODE('mp4v')
						|| m_CodecInfo.m_iFourCC == FOUR_CHAR_CODE('svq3') )) {*/
	if (sampleTable && m_iMode != QTMOVIE_MODE_MTASK) {
		bUseSampleTable = true;

		//TimeValue64 start = GetMediaDisplayStartTime(m_vMedia);
		//TimeValue64 decodetime = GetMediaAdvanceDecodeTime(m_vMedia);

		TimeValue64 myCurTime64 = 0, myDur64 = 0, myDur64_First;
		UInt32 mSize;

		myFrameCount = 0;
		// Loop through all the frames (sample) in the movie
		SInt64 numSamples = QTSampleTableGetNumberOfSamples(sampleTable);
		for (i = 1; i <= numSamples; i++) {
			// Get the DisplayOffset and flags for this frame
			offset1 = QTSampleTableGetDisplayOffset(sampleTable, i);
			flags = QTSampleTableGetSampleFlags(sampleTable, i);
			mSize = QTSampleTableGetDataSizePerSample(sampleTable, i);

			// Get the MediaDisplay Time
			SampleNumToMediaDisplayTime(m_vMedia, i, &myCurTime64, &myDur64);

			myMpegFrameInfo.timePlay = myCurTime64;
			myMpegFrameInfo.durPlay = myDur64;
			myMpegFrameInfo.idx = i - 1;
			myMpegFrameInfo.mOffset = offset1;
			myMpegFrameInfo.mFlags = flags;
			myMpegFrameInfo.mSize = mSize;

			// Get the MediaDecode Time
			SampleNumToMediaDecodeTime(m_vMedia, i, &myCurTime64, &myDur64);

			myMpegFrameInfo.timeDecode = myCurTime64;
			myMpegFrameInfo.durDecode = myDur64;
			//myMpegFrameInfo.mFrameType = !(flags & mediaSampleNotSync);
			if (! (flags & mediaSampleNotSync) || (flags & mediaSampleDoesNotDependOnOthers))
				myMpegFrameInfo.mFrameType = 0; // #define QTFrameType_I 0
			else if (flags & mediaSampleNotSync)
				myMpegFrameInfo.mFrameType = 1; // #define QTFrameType_P 1
			else if (flags & mediaSampleNotSync)
				myMpegFrameInfo.mFrameType = 2; // #define QTFrameType_B 2

			if (! (flags & mediaSampleNotSync) || (flags & mediaSampleDoesNotDependOnOthers))
				// The frame is an IFrame, store it's idx
				m_vKeyframes.push_back(i-1);
			else
				bIFrameOnly = false;

			// Store the frame in our Display frame vector
			m_vFrameInfoDisplay.push_back(myMpegFrameInfo);

			if (i == 1) {
				// USe the values of the 1st frame as global values for the movie
				myDur64_First = myMpegFrameInfo.durPlay;
				m_SampleDuration = myMpegFrameInfo.durDecode;
			}

//????WHY do this next step???
			// Modify the idx for the Decode frame vector
			myMpegFrameInfo.idx = (myMpegFrameInfo.timePlay / myDur64_First) - 1;

			// Store the frame in our Decode frame vector
			m_vFrameInfoDecode.push_back(myMpegFrameInfo);

			// Increment the frame count
			myFrameCount++;
		}

		// Sort the frame vectors by display time
		sort(m_vFrameInfoDisplay.begin(), m_vFrameInfoDisplay.end(), SortByDisplayTime);
		sort(m_vFrameInfoDecode.begin(), m_vFrameInfoDecode.end(), SortByDisplayTime);

//????WHY do this next step???
		// Now reset the idx of the sorted Decode vector
		// so that the idx is the playFframe number
		for (i = 0; i < m_vFrameInfoDecode.size(); i++) {
			m_vFrameInfoDecode[i].idx = i;
		}

		// Sort by decode time
		sort(m_vFrameInfoDecode.begin(), m_vFrameInfoDecode.end(), SortByDecodeTime);
	} else {
		// The movie does not contain a sample table so use GetMovieNextInterestingTime
		bool bVfr = false;
		short myFlags;
		OSType myTypes[1];
		myTypes[0] = VisualMediaCharacteristic;
		TimeValue myCurTime = 0, myDur = 0; //kBogusStartingTime
		myFlags = nextTimeStep + nextTimeEdgeOK;

		// Get the first tiome value
		GetMovieNextInterestingTime(m_Movie,
			myFlags, 1, myTypes, (TimeValue)0, fixed1, &myCurTime, &myDur);

		// Use the first value for our global val
		m_SampleDuration = myDur;

		myFlags = nextTimeStep; //nextTimeMediaSample nextTimeStep nextTimeSyncSample

		// Loop through movie, getting sample times
		while (myCurTime >= 0) {
			int count = 1;
			if (myDur > m_SampleDuration) {
				// The current frame durations is not the same as the first frame
				// This means the movie is variable frame rate
				bVfr = true;

				// Using the 1st frame sample duration, how many frames
				// belong to this one sample?
				count = 0.5 + ((float)myDur / m_SampleDuration);
			}

			// Now loop through the frame count and add them to our Display frame vector
			for (i = 0; i < count; i++) {
				myFrameCount++;
				/*
				myFrameInfo.timePlay = myCurTime;
				myFrameInfo.durPlay = myDur;
				m_vFrameInfo.push_back(myFrameInfo);
				*/
				myMpegFrameInfo.timePlay = myCurTime;
				myMpegFrameInfo.durPlay = myDur;
				myMpegFrameInfo.idx = myFrameCount;
				m_vFrameInfoDisplay.push_back(myMpegFrameInfo);
			}

			GetMovieNextInterestingTime(m_Movie,
				myFlags, 1, myTypes, myCurTime, fixed1, &myCurTime, &myDur);
		} // end while (myCurTime >= 0) {

		if (bVfr) {
			m_vMediaFrameCount = myFrameCount;
			// if (QTVersion7) {
			m_vMediaSampleDuration =
				(float)GetMediaDecodeDuration(m_vMedia) / (float)m_vMediaFrameCount;
			m_SampleDuration =
				(float)GetMediaDecodeDuration(m_vMedia) / (float)m_vMediaFrameCount;
			//if (!QTVersion7) {
			//m_vMediaSampleDuration = (float)m_vMediaDur / (float)m_vMediaFrameCount;
			//m_SampleDuration = (float)m_vMediaDur / (float)m_vMediaFrameCount;
		}
	} // end else (sampleTable) {

	// Make sure there is not an empty frame in m_vFrameInfoDisplay
	if (m_vMediaFrameCount > 1 && myFrameCount < 2) {
		//m_vFrameInfo.clear();
		m_vFrameInfoDisplay.clear();
		m_vMovieFrameCount = m_vMediaFrameCount;
	} else {
		//m_vMediaFrameCount = myCount;
		m_vMovieFrameCount = myFrameCount;
	}

	// Setup frame byte size params, using a width that is aligned on m_iMod bytes
	float mMult = 3.0;
	int m_iMod = 16, m_iModWidth;
	if (m_pixelFormat == k32BGRAPixelFormat) {
		mMult = 4.0;
		m_iMod = 4;
	} else if (m_pixelFormat == kYUVSPixelFormat) {
		mMult = 2.0;
		m_iMod = 8;
	} else if (m_pixelFormat == kYUV420PixelFormat || m_pixelFormat == kMpegYUV420CodecType) {
		mMult = 12.0/8.0;
		m_iMod = 16;
	}

	if ((m_vTrackFrame.right % m_iMod) != 0) {
		m_iModWidth = (floor(1.0 * m_vTrackFrame.right / m_iMod) + 1) * m_iMod;
	} else
		m_iModWidth = m_vTrackFrame.right;

	m_iModRowsize = m_iModWidth * mMult;
	m_iRowsize = m_vTrackFrame.right * mMult;

	/*if ((m_Rowsize % 16) != 0) {
		m_iModRowsize = (floor(1.0 * m_Rowsize / 16) + 1) * 16;
	} else
		m_iModRowsize = m_Rowsize;*/

	// Make this a little bigger than RGBA
	maxCompressedSize = m_vTrackFrame.right * m_vTrackFrame.bottom * 4.5;

	// Set this here for ICMDecompression
	if (bContainsDisplayOffsets)
		nextIFrame = 0;
	else
		nextIFrame = m_vMovieFrameCount + 1;

	CompressorComponent compressor;
	DecompressorComponent decompressor = NULL;

	// Setup shared parameters used in all modes except for ICM
	if (m_iMode != QTMOVIE_MODE_ICM && m_iMode != QTMOVIE_MODE_VFW) {
		err = InitDecodeShared();
		if (err != noErr)
			goto TerminateQuickTime;
	}

	err = InitDecodeFindCodec(compressor, decompressor);
	if (err != noErr)
		goto TerminateQuickTime;

	if (m_iMode == QTMOVIE_MODE_MTASK) {
		err = InitDecodeMTask();
		if (err != noErr)
			goto TerminateQuickTime;
	} else if (m_iMode == QTMOVIE_MODE_RAW) {
		err = InitDecodeDataPointers();
		if (err != noErr)
			goto TerminateQuickTime;

		err = InitDecodeRaw();
		if (err != noErr)
			goto TerminateQuickTime;
	} else if (m_iMode == QTMOVIE_MODE_DESQNC) {
		err = InitDecodeDataPointers();
		if (err != noErr)
			goto TerminateQuickTime;

		err = InitDecodeDeSqnc(decompressor);
		if (err != noErr)
			goto TerminateQuickTime;
	} else if (m_iMode == QTMOVIE_MODE_ICM) {
		err = InitDecodeICM();
		if (err != noErr)
			goto TerminateQuickTime;
	} else if (m_iMode == QTMOVIE_MODE_VFW) {
		err = InitDecodeVFW(strFourCC);
		if (err != noErr)
			goto TerminateQuickTime;
	} else {
		// Error, not a valid mode
	}

SETUP_AUDIO:

	m_AudioExtractor = NULL;
	if (mAudio == 1 && m_aTrackCount > 0) { // && bAudioExtractionApi) {
		//QTMOVIE_QT7
		if (! SetUpMovieAudioExtraction())
			goto TerminateQuickTime;
	}

	//Must Clear this for some reason
	//When I have 6 channel audio, the first call to IsFrameInBuffer
	//returns m_vFrameBuffer.size() = 6293
	m_vFrameBuffer.clear();

	return S_OK;

TerminateQuickTime:

	//QTFreeMemory();
	return E_FAIL;
}

int CQTMovieDec::ReadVideoFrame(long frameNum, byte *pData, int dst_pitch, int *rawSize)
{
	OSErr err = noErr;
	int retVal = 0;

	TimeValue time, frameNumTime, sampleTime = 0, durationPerSample = 0;
	long sampleDescriptionIndex = 0, numberOfSamples = 1, maxNumberOfSamples = 1;
	long frameNumCurrent = frameNum, curFrame = -1;
	bool bFoundIFrame = true;
	short sampleflags = 0;
	long m_lBytesReturned = 0;

	if (frameNum >= m_vFrameInfoDisplay.size()) {
		// 2005-01-03, GetMovieNextInterestingTime and SampleTable failed to get frame times
		// fall back to manual method of determining frame time
		frameNumTime = (frameNum * m_vTrackIncrement) + m_vTrackStartTime;
		time = TrackTimeToMediaTime(frameNumTime, m_vTrack);
		//sprintf(strStatus, "Frame, frameNum=%d, time=%d, frameNumTime=%d", frameNum, time, frameNumTime);
	} else if (m_vFrameInfoDisplay.size() > 0) {
		frameNumTime = m_vFrameInfoDisplay[frameNum].timePlay; //m_vFrameInfoDisplay
		if (bUseSampleTable || m_iMode == QTMOVIE_MODE_MTASK)
			time = frameNumTime;
		else
			time = TrackTimeToMediaTime(frameNumTime, m_vTrack);
		//sprintf(strStatus, "Vector, frameNum=%d, time=%d, frameNumTime=%d, bUseSampleTable=%d",
		//	frameNum, time, frameNumTime, bUseSampleTable);
	}

	if (m_iMode  == QTMOVIE_MODE_MTASK) {
		retVal = FrameDecodeMTask(frameNumTime, pData, dst_pitch, rawSize);
	} else if (m_iMode  == QTMOVIE_MODE_RAW) {
		retVal = FrameDecodeRaw();
	} else if (m_iMode  == QTMOVIE_MODE_DESQNC) {
		retVal = FrameDecodeDeSqnc(frameNum, frameNumTime, pData, dst_pitch, rawSize);
	} else if (m_iMode  == QTMOVIE_MODE_ICM) {
		/*bool bNeedNextI = false;
		if (bContainsDisplayOffsets
				&& abs(frameNum - m_lPrevFrameNum) > 1
				&& frameNum+1 > nextIFrame) {
			bNeedNextI = true;
		}*/

		/*if (frameNum > nextIFrame) {
			TimeValue64 syncDecodeTime, targetDecodeTime;

			SampleNumToMediaDecodeTime(m_vMedia, frameNum, &targetDecodeTime, NULL);

			GetMediaNextInterestingDecodeTime( m_vMedia,
				nextTimeSyncSample | nextTimeEdgeOK,
				targetDecodeTime,
				-fixed1,
				&syncDecodeTime,
				NULL );
			MediaDisplayTimeToSampleNum(m_vMedia, syncDecodeTime, &nextIFrame, NULL, NULL);*/

		if (abs(frameNum - m_lPrevFrameNum) > 20 && frameNum != 0) {
			retVal = FrameDecodeICM(frameNum - 1, frameNumTime, pData, dst_pitch, rawSize);
			//retVal = FrameDecodeICM(nextIFrame + 1, frameNumTime, pData, dst_pitch);
			retVal = FrameDecodeICM(frameNum, frameNumTime, pData, dst_pitch, rawSize);
		} else
			retVal = FrameDecodeICM(frameNum, frameNumTime, pData, dst_pitch, rawSize);

		/*if (bNeedNextI) {
			TimeValue64 syncDecodeTime, targetDecodeTime;

			SampleNumToMediaDecodeTime(m_vMedia, frameNum+2, &targetDecodeTime, NULL);

			GetMediaNextInterestingDecodeTime( m_vMedia,
				nextTimeSyncSample | nextTimeEdgeOK,
				targetDecodeTime,
				fixed1,
				&syncDecodeTime,
				NULL );
			MediaDisplayTimeToSampleNum(m_vMedia, syncDecodeTime, &nextIFrame, NULL, NULL);
		}*/
	} else if (m_iMode  == QTMOVIE_MODE_VFW) {
		retVal = FrameDecodeDeSqnc(frameNum, frameNumTime, pData, dst_pitch, rawSize); //FrameDecodeVFW
	}

	return retVal;
}