//	Quicktime Import Plugin for VirtualDub
//	Copyright (C) 2007 Josh Harris (tateu)
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "QTMovieDec.h"
#include "vdplugin.h"
#include "vdinputdriver.h"
#include <string.h>
#include <vector>
#include <math.h>

#include "resource.h"

#ifndef MAX_DRIVE
#define MAX_DRIVE 10
#define MAX_DIR MAX_PATH
#define MAX_FNAME MAX_PATH
#define MAX_EXT MAX_PATH
#endif

#define VDRoundToLong(x) (long)(x + 0.5)
#define VDRoundToInt(x) (int)(x + 0.5)

// Copied From VirtualDub, gui.cpp
void ticks_to_str(char *dst, size_t bufsize, uint32 ticks) {
	int sec, min, hr, day;

	ticks /= 1000;
	sec	= ticks %  60; ticks /=  60;
	min	= ticks %  60; ticks /=  60;
	hr	= ticks %  24; ticks /=  24;
	day	= ticks;

	if (day)
		_snprintf(dst, bufsize, "%d:%02d:%02d:%02d",day,hr,min,sec);
	else if (hr)
		_snprintf(dst, bufsize, "%d:%02d:%02d",hr,min,sec);
	else
		_snprintf(dst, bufsize, "%d:%02d",min,sec);
}
//

unsigned long codecQuality[6] = {codecMinQuality, codecLowQuality, codecNormalQuality,
		codecHighQuality, codecMaxQuality, codecLosslessQuality };

//TEMP
int pixFormat;
///////////////////////////////////////////////////////////////////////////////

HMODULE g_hInst;

BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved) {
	if (fdwReason == DLL_PROCESS_ATTACH) {
		g_hInst = (HMODULE)hinstDLL;
	}
	return TRUE;
}

extern "C" long _InterlockedExchangeAdd(volatile long *p, long v);
#pragma intrinsic(_InterlockedExchangeAdd)

namespace {
	template<class T> class vdxunknown : public T {
	public:
		vdxunknown() : mRefCount(0) {}
		vdxunknown(const vdxunknown<T>& src) : mRefCount(0) {}		// do not copy the refcount
		virtual ~vdxunknown() {}

		vdxunknown<T>& operator=(const vdxunknown<T>&) {}			// do not copy the refcount

		virtual int VDXAPIENTRY AddRef() {
			return _InterlockedExchangeAdd(&mRefCount, 1) + 1;
		}

		virtual int VDXAPIENTRY Release() {
			long rc = _InterlockedExchangeAdd(&mRefCount, -1) - 1;
			if (!mRefCount) {
				mRefCount = 1;
				delete this;
				return 0;
			}

			return rc;
		}

		virtual void *VDXAPIENTRY AsInterface(uint32 iid) {
			if (iid == T::kIID)
				return static_cast<T *>(this);

			if (iid == IVDXUnknown::kIID)
				return static_cast<IVDXUnknown *>(this);

			return NULL;
		}

	protected:
		volatile long	mRefCount;
	};
}

///////////////////////////////////////////////////////////////////////////////

struct QuicktimeHeader {
	//uint32	mSignature;
	//uint32	mVersion;
	uint32	mWidth;
	uint32	mHeight;
	uint8	mDepth;
	sint64  mFrames;
	uint32	mFrameRateNumerator;
	uint32	mFrameRateDenominator;
	//uint32	mUnused0;
	//uint64	mIndexOffset;
	//uint64	mFrameDataOffset;

	uint16 mFormatTag;
	uint16 mChannels;
	uint32 mSamplesPerSec;
	uint32 mAvgBytesPerSec;
	uint16 mBlockAlign;
	uint16 mBitsPerSample;
	uint16 mExtraSize;
	VDXFraction mSampleRate;
	sint64 mSampleCount;

};

/*struct QuicktimeIndexEntry {
	uint64	mFrameOffset;
	uint32	mSize;
	uint8	mFrameType;
	uint8	mUnused[3];
};*/

class QuicktimeData {
public:
	CQTMovieDec *m_QTMovie, *m_QTMovieRGB32;
	int _vtrack, _atrack, _color, _raw, _audio, _dither;
	int m_iMode, quality;
	float m_fGamma;
	char strVFW[5];
	QTCodecInfo mCodec, m_DecoderCodec;

	QuicktimeHeader	mHeader;
	//std::vector<QuicktimeIndexEntry> mIndex;
	vector<long>				m_vKeyframes;
	vector<mpegFrameInfo>		m_vFrameInfoDisplay, m_vFrameInfoDecode;
	int frameSize;
	int pixFormat;
};

///////////////////////////////////////////////////////////////////////////////

class VDVideoDecoderPluginQuicktime : public vdxunknown<IVDXVideoDecoder> {
public:
	VDVideoDecoderPluginQuicktime(const QuicktimeData& data);
	~VDVideoDecoderPluginQuicktime();

	const void *VDXAPIENTRY DecodeFrame(const void *inputBuffer, uint32 data_len, bool is_preroll, sint64 streamFrame, sint64 targetFrame);
	uint32		VDXAPIENTRY GetDecodePadding();
	void		VDXAPIENTRY Reset();
	bool		VDXAPIENTRY IsFrameBufferValid();
	const VDXPixmap& VDXAPIENTRY GetFrameBuffer();
	bool		VDXAPIENTRY SetTargetFormat(int format, bool useDIBAlignment);
	bool		VDXAPIENTRY SetDecompressedFormat(const VDXBITMAPINFOHEADER *pbih);

	const void *VDXAPIENTRY GetFrameBufferBase();
	bool		VDXAPIENTRY IsDecodable(sint64 sample_num);

protected:
	sint64	mLastFrame;

	VDXPixmap	mPixmap;

	unsigned char *mFrameBuffer;

	//VDXPixmap			mYCbCrBufferBidir;
	//VDXPixmap			mYCbCrBufferPrev;
	//VDXPixmap			mYCbCrBufferNext;
	//std::vector<uint8>	mYCbCrBufferMemory;
	sint64				mPrevFrame;
	sint64				mNextFrame;

	const QuicktimeData&	mData;
};

VDVideoDecoderPluginQuicktime::VDVideoDecoderPluginQuicktime(const QuicktimeData& data)
	: mData(data)
{
	mFrameBuffer = (unsigned char *)malloc(mData.mHeader.mWidth * mData.mHeader.mHeight * 4);
	//mFrameBuffer.resize(mData.mHeader.mWidth * mData.mHeader.mHeight * pitch_Mult);
	memset(mFrameBuffer, 0, mData.mHeader.mWidth * mData.mHeader.mHeight * 4);

	mPixmap.data			= mFrameBuffer;
	mPixmap.palette			= NULL;
	mPixmap.w				= mData.mHeader.mWidth;
	mPixmap.h				= mData.mHeader.mHeight;
	//mPixmap.format			= nsVDXPixmap::kPixFormat_XRGB8888;
	mPixmap.pitch			= mData.mHeader.mWidth * 4;
	mPixmap.data2			= NULL;
	mPixmap.pitch2			= 0;
	mPixmap.data3			= NULL;
	mPixmap.pitch3			= 0;
}

VDVideoDecoderPluginQuicktime::~VDVideoDecoderPluginQuicktime()
{
	free(mFrameBuffer);
}

const void *VDVideoDecoderPluginQuicktime::DecodeFrame(const void *inputBuffer, uint32 data_len, bool is_preroll, sint64 streamFrame, sint64 targetFrame)
{
	if (mData.m_vFrameInfoDisplay.size() == 0)
		return mFrameBuffer;

	if (data_len) { // && (mLastFrame != targetFrame || mLastFrame < 0)) {
		if (! is_preroll) {
			//memcpy(mPixmap.data, inputBuffer, data_len);
			if ((mData._color == 2 && mPixmap.format != nsVDXPixmap::kPixFormat_YUV422_YUYV)
					|| (mData._color == 3 && mPixmap.format != nsVDXPixmap::kPixFormat_YUV420_Planar))
				memcpy(mFrameBuffer, inputBuffer, mData.mHeader.mWidth * mData.mHeader.mHeight * 4);
			else
				memcpy(mFrameBuffer, inputBuffer, mData.frameSize);
			mLastFrame = targetFrame;
		}
	}

	return mFrameBuffer;
}

uint32 VDVideoDecoderPluginQuicktime::GetDecodePadding()
{
	return 0;
}

void VDVideoDecoderPluginQuicktime::Reset()
{
	mLastFrame = -1;
	mPrevFrame = -1;
	mNextFrame = -1;
}

bool VDVideoDecoderPluginQuicktime::IsFrameBufferValid()
{
	return mLastFrame >= 0;
}

const VDXPixmap& VDVideoDecoderPluginQuicktime::GetFrameBuffer()
{
	return mPixmap;
}

bool VDVideoDecoderPluginQuicktime::SetTargetFormat(int format, bool useDIBAlignment)
{
	float pitch_Mult;

	if (format == 0) {
		if (mData._color == 0)
			format = nsVDXPixmap::kPixFormat_RGB888;
		else if (mData._color == 1)
			format = nsVDXPixmap::kPixFormat_XRGB8888;
		else if (mData._color == 2)
			format = nsVDXPixmap::kPixFormat_YUV422_YUYV;
			// kPixFormat_YUV422_UYVY kPixFormat_YUV422_YUYV
		else if (mData._color == 3)
			format = nsVDXPixmap::kPixFormat_YUV420_Planar;
	} else if (mData._color == 0 && format != nsVDXPixmap::kPixFormat_RGB888) {
		return false;
	} else if (mData._color == 1 && format != nsVDXPixmap::kPixFormat_XRGB8888) {
		return false;
	} else if (mData._color > 1 && format == nsVDXPixmap::kPixFormat_RGB888) {
		// This forces it to move onto RGB32
		return false;
	} else if (mData._color == 2 && format != nsVDXPixmap::kPixFormat_YUV422_YUYV) {
		//return false;
	} else if (mData._color == 3 && format != nsVDXPixmap::kPixFormat_YUV420_Planar) {
		//return false;
	} else if (
			(format != nsVDXPixmap::kPixFormat_RGB888) &&
			(format != nsVDXPixmap::kPixFormat_XRGB8888) &&
			(format != nsVDXPixmap::kPixFormat_YUV422_YUYV) &&
			(format != nsVDXPixmap::kPixFormat_YUV420_Planar)
		) {
		return false;
	}/* else if (
			(mData._color == 0 && format != nsVDXPixmap::kPixFormat_RGB888) ||
			(mData._color == 1 && format != nsVDXPixmap::kPixFormat_XRGB8888) ||
			(mData._color == 2 && format != nsVDXPixmap::kPixFormat_YUV422_YUYV) ||
			(mData._color == 3 && format != nsVDXPixmap::kPixFormat_YUV420_Planar)
		) {
		return false;
	}*/

	pixFormat = format;

	mPixmap.data			= mFrameBuffer;
	mPixmap.palette			= NULL;
	mPixmap.format			= format;
	mPixmap.w				= mData.mHeader.mWidth;
	mPixmap.h				= mData.mHeader.mHeight;

	if (format == nsVDXPixmap::kPixFormat_RGB888) {
		pitch_Mult = 3;

		mPixmap.pitch			= mData.mHeader.mWidth * pitch_Mult;
		mPixmap.data2			= NULL;
		mPixmap.pitch2			= 0;
		mPixmap.data3			= NULL;
		mPixmap.pitch3			= 0;
	} else if (format == nsVDXPixmap::kPixFormat_XRGB8888) {
		pitch_Mult = 4;

		mPixmap.pitch			= mData.mHeader.mWidth * pitch_Mult;
		mPixmap.data2			= NULL;
		mPixmap.pitch2			= 0;
		mPixmap.data3			= NULL;
		mPixmap.pitch3			= 0;
	} else if (format == nsVDXPixmap::kPixFormat_YUV422_YUYV) {
		pitch_Mult = 2;

		mPixmap.pitch			= mData.mHeader.mWidth * pitch_Mult;
		mPixmap.data2			= NULL;
		mPixmap.pitch2			= 0;
		mPixmap.data3			= NULL;
		mPixmap.pitch3			= 0;
	} else if (format == nsVDXPixmap::kPixFormat_YUV420_Planar) {
		pitch_Mult = 1.5;

		mPixmap.pitch			= mData.mHeader.mWidth; // * pitch_Mult;
		mPixmap.data2			= mFrameBuffer
								+ mData.mHeader.mWidth * mData.mHeader.mHeight
								+ (mData.mHeader.mWidth >> 1) * (mData.mHeader.mHeight >> 1);
		mPixmap.pitch2			= mData.mHeader.mWidth >> 1;
		mPixmap.data3			= mFrameBuffer + mData.mHeader.mWidth * mData.mHeader.mHeight;
		mPixmap.pitch3			= mData.mHeader.mWidth >> 1;

		/*mPixmap.data2			= mFrameBuffer
									+ mData.mHeader.mWidth * mData.mHeader.mHeight;
		mPixmap.pitch2			= mData.mHeader.mWidth / 4;
		mPixmap.data3			= mFrameBuffer
									+ (mData.mHeader.mWidth * mData.mHeader.mHeight)
									+ (mData.mHeader.mWidth * mData.mHeader.mHeight / 4)
									+ (mData.mHeader.mWidth * mData.mHeader.mHeight / 4);
		mPixmap.pitch3			= mData.mHeader.mWidth / 4;*/
	}

	if (useDIBAlignment && format != nsVDXPixmap::kPixFormat_YUV420_Planar
			&& format != nsVDXPixmap::kPixFormat_YUV422_YUYV) {
		mPixmap.data	= (char *)mPixmap.data + mPixmap.pitch*(mPixmap.h - 1);
		mPixmap.pitch	= -mPixmap.pitch;
	}

	int order;
	if ((mData._color == 2 && format != nsVDXPixmap::kPixFormat_YUV422_YUYV)
			|| (mData._color == 3 && format != nsVDXPixmap::kPixFormat_YUV420_Planar)) {
		if (mData.m_iMode == QTMOVIE_MODE_VFW)
			order = 1;
		else
			order = 0;
		mData.m_QTMovie->SetRowOrder(order);
		if (mData.m_QTMovieRGB32)
			mData.m_QTMovieRGB32->SetRowOrder(order);
	} else {
		if (mData._color > 1 || mData.m_iMode == QTMOVIE_MODE_VFW)
			order = 1;
		else
			order = 0;
		mData.m_QTMovie->SetRowOrder(order);
		if (mData.m_QTMovieRGB32)
			mData.m_QTMovieRGB32->SetRowOrder(order);
	}

	Reset();

	return true;
}

bool VDVideoDecoderPluginQuicktime::SetDecompressedFormat(const VDXBITMAPINFOHEADER *pbih)
{
	return false;
}

const void *VDVideoDecoderPluginQuicktime::GetFrameBufferBase()
{
	return mFrameBuffer;
}

bool VDVideoDecoderPluginQuicktime::IsDecodable(sint64 sample_num64)
{
	if (sample_num64 < mData.m_vFrameInfoDisplay.size())
		return true;
	else
		return false;

/*
	uint32 sample_num = (uint32)sample_num64;
	//uint8 frameType = mData.m_vFrameInfoDisplay[sample_num].mFrameType;
	uint8 frameType = mData.m_vFrameInfoDisplay[sample_num].mFrameType;

	switch(frameType) {
		// B frames require two predictive frames
		case 2:
			for(;;) {
				--sample_num;

				if (mData.m_vFrameInfoDisplay[sample_num].mFrameType < 2)
					break;
			}

			if (mPrevFrame != sample_num && mNextFrame != sample_num)
				return false;

			// fall through

		case 1:
			// P and B frames require one predictive frame
			for(;;) {
				--sample_num;

				if (mData.m_vFrameInfoDisplay[sample_num].mFrameType < 2)
					break;
			}

			if (mPrevFrame != sample_num && mNextFrame != sample_num)
				return false;

		// I frames are always decodable
		case 0:
		default:
			return true;
	}

	return true;
*/
}

///////////////////////////////////////////////////////////////////////////////

class VDVideoDecoderModelPluginQuicktime : public vdxunknown<IVDXVideoDecoderModel> {
public:
	VDVideoDecoderModelPluginQuicktime(const QuicktimeData& data);
	~VDVideoDecoderModelPluginQuicktime();

	void	VDXAPIENTRY Reset();
	void	VDXAPIENTRY SetDesiredFrame(sint64 frame_num);
	sint64	VDXAPIENTRY GetNextRequiredSample(bool& is_preroll);
	int		VDXAPIENTRY GetRequiredCount();

protected:
	sint64	mLastPrevFrame;
	sint64	mLastNextFrame;
	sint64	mLastBidirFrame;
	sint64	mTargetFrame;
	sint64	mNextFrame;

	const QuicktimeData& mData;
};

VDVideoDecoderModelPluginQuicktime::VDVideoDecoderModelPluginQuicktime(const QuicktimeData& data)
	: mData(data)
{
}

VDVideoDecoderModelPluginQuicktime::~VDVideoDecoderModelPluginQuicktime()
{
}

void VDVideoDecoderModelPluginQuicktime::Reset()
{
	mLastPrevFrame = -1;
	mLastNextFrame = -1;
	mLastBidirFrame = -1;
}

void VDVideoDecoderModelPluginQuicktime::SetDesiredFrame(sint64 frame_num)
{
	mTargetFrame = frame_num;

	if (mLastPrevFrame == frame_num || mLastNextFrame == frame_num || mLastBidirFrame == frame_num) {
		mNextFrame = -1;
		return;
	}

	uint8 frameType = 0; //mData.m_vFrameInfoDisplay[(uint32)frame_num].mFrameType;

	// I-frame is trivial
	if (frameType == 0) {
		mNextFrame = frame_num;
		return;
	}

/*
	// P-frame is more annoying
	if (frameType == 1) {
		// find previous IP frame
		sint64 prevP = frame_num ? frame_num - 1 : 0;
		while(prevP > 0 && mData.m_vFrameInfoDisplay[(uint32)prevP].mFrameType == 2)
			--prevP;

		// find previous I frame
		sint64 prevI = prevP;
		while(prevI > 0 && mData.m_vFrameInfoDisplay[(uint32)prevI].mFrameType != 0)
			--prevI;

		// what's the nearest frame that we have?
		sint64 validPrev = mLastPrevFrame >= prevI && mLastPrevFrame < frame_num ? mLastPrevFrame : -1;
		sint64 validNext = mLastNextFrame >= prevI && mLastNextFrame < frame_num ? mLastNextFrame : -1;

		if (validPrev > validNext)
			mNextFrame = validPrev + 1;
		else if (validNext > validPrev)
			mNextFrame = validNext + 1;
		else
			mNextFrame = prevI;
		return;
	}

	// B-frame is very annoying

	// find previous IP frame
	sint64 prevP2 = frame_num;
	while(prevP2 > 0 && mData.m_vFrameInfoDisplay[(uint32)prevP2].mFrameType == 2)
		--prevP2;

	// find previous IP frame before that
	sint64 prevP1 = prevP2 ? prevP2 - 1 : 0;
	while(prevP1 > 0 && mData.m_vFrameInfoDisplay[(uint32)prevP1].mFrameType == 2)
		--prevP1;

	// If we have next & prev, we can just decode off that.
	// If we have only prev, then we can decode from that.
	// Otherwise, we need to decode from the nearest reference to prev.

	bool havePrev = (mLastPrevFrame == prevP1 || mLastNextFrame == prevP1);
	bool haveNext = (mLastPrevFrame == prevP2 || mLastNextFrame == prevP2);

	if (havePrev) {
		if (haveNext)
			mNextFrame = prevP2 + 1;
		else
			mNextFrame = prevP1 + 1;
		return;
	}

	// Okay, we need to decode the prev frame. Find the nearest I frame.
	sint64 prevI = prevP1;
	while(prevI > 0 && mData.m_vFrameInfoDisplay[(uint32)prevI].mFrameType != 0)
		--prevI;

	// what's the nearest frame that we have?
	sint64 validPrev = mLastPrevFrame >= prevI && mLastPrevFrame < prevP1 ? mLastPrevFrame : -1;
	sint64 validNext = mLastNextFrame >= prevI && mLastNextFrame < prevP1 ? mLastNextFrame : -1;

	if (validPrev > validNext) {
		std::swap(mLastPrevFrame, mLastNextFrame);
		mNextFrame = validPrev + 1;
	} else if (validNext > validPrev) {
		std::swap(mLastNextFrame, mLastPrevFrame);
		mNextFrame = validNext + 1;
	} else {
		mNextFrame = prevI;
	}
*/
}

sint64 VDVideoDecoderModelPluginQuicktime::GetNextRequiredSample(bool& is_preroll)
{
	if (mNextFrame < 0 || mNextFrame == mTargetFrame) {
		is_preroll = false;
		sint64 frame = mNextFrame;
		mNextFrame = -1;
		return frame;
	}

	return -1;

/*
	is_preroll = true;
	sint64 frame;
	
	do {
		frame = mNextFrame++;
	} while(frame != mTargetFrame && mData.m_vFrameInfoDisplay[(uint32)frame].mFrameType == 2);

	if (mData.m_vFrameInfoDisplay[(uint32)frame].mFrameType == 2) {
		mLastBidirFrame = frame;
	} else {
		if ((uint64)(frame - mLastNextFrame) < (uint64)(frame - mLastPrevFrame))
			std::swap(mLastPrevFrame, mLastNextFrame);

		mLastNextFrame = frame;
	}

	if (frame == mTargetFrame) {
		mNextFrame = -1;
		is_preroll = false;
	}

	return frame;
*/
}

int VDVideoDecoderModelPluginQuicktime::GetRequiredCount()
{
	return mNextFrame == -1 ? 0 : 1;
}

///////////////////////////////////////////////////////////////////////////////

class VDVideoSourcePluginQuicktime : public vdxunknown<IVDXStreamSource>, public IVDXVideoSource {
public:
	VDVideoSourcePluginQuicktime(QuicktimeData& data);
	~VDVideoSourcePluginQuicktime();

	int VDXAPIENTRY AddRef();
	int VDXAPIENTRY Release();
	void *VDXAPIENTRY AsInterface(uint32 iid);

	void		VDXAPIENTRY GetStreamSourceInfo(VDXStreamSourceInfo&);
	bool		VDXAPIENTRY Read(sint64 lStart, uint32 lCount, void *lpBuffer, uint32 cbBuffer, uint32 *lBytesRead, uint32 *lSamplesRead);

	const void *VDXAPIENTRY GetDirectFormat();
	int			VDXAPIENTRY GetDirectFormatLen();

	ErrorMode VDXAPIENTRY GetDecodeErrorMode();
	void VDXAPIENTRY SetDecodeErrorMode(ErrorMode mode);
	bool VDXAPIENTRY IsDecodeErrorModeSupported(ErrorMode mode);

	bool VDXAPIENTRY IsVBR();
	sint64 VDXAPIENTRY TimeToPositionVBR(sint64 us);
	sint64 VDXAPIENTRY PositionToTimeVBR(sint64 samples);

	void VDXAPIENTRY GetVideoSourceInfo(VDXVideoSourceInfo& info);

	bool VDXAPIENTRY CreateVideoDecoderModel(IVDXVideoDecoderModel **ppModel);
	bool VDXAPIENTRY CreateVideoDecoder(IVDXVideoDecoder **ppDecoder);

	void		VDXAPIENTRY GetSampleInfo(sint64 sample_num, VDXVideoFrameInfo& frameInfo);

	bool		VDXAPIENTRY IsKey(sint64 lSample);

	sint64		VDXAPIENTRY GetFrameNumberForSample(sint64 sample_num);
	sint64		VDXAPIENTRY GetSampleNumberForFrame(sint64 display_num);
	sint64		VDXAPIENTRY GetRealFrame(sint64 display_num);

	sint64		VDXAPIENTRY GetSampleBytePosition(sint64 sample_num);

protected:
	QuicktimeData&	mData;
	sint64				mPrevFrame;
};

VDVideoSourcePluginQuicktime::VDVideoSourcePluginQuicktime(QuicktimeData& data)
	: mData(data)
{
	mPrevFrame = -1;
}

VDVideoSourcePluginQuicktime::~VDVideoSourcePluginQuicktime()
{
}

int VDVideoSourcePluginQuicktime::AddRef()
{
	return vdxunknown<IVDXStreamSource>::AddRef();
}

int VDVideoSourcePluginQuicktime::Release()
{
	return vdxunknown<IVDXStreamSource>::Release();
}

void *VDXAPIENTRY VDVideoSourcePluginQuicktime::AsInterface(uint32 iid)
{
	if (iid == IVDXVideoSource::kIID)
		return static_cast<IVDXVideoSource *>(this);

	return vdxunknown<IVDXStreamSource>::AsInterface(iid);
}

void VDXAPIENTRY VDVideoSourcePluginQuicktime::GetStreamSourceInfo(VDXStreamSourceInfo& srcInfo)
{
	srcInfo.mSampleRate.mNumerator = mData.mHeader.mFrameRateNumerator;
	srcInfo.mSampleRate.mDenominator = mData.mHeader.mFrameRateDenominator;
	srcInfo.mSampleCount = mData.mHeader.mFrames;
}

bool VDVideoSourcePluginQuicktime::Read(sint64 lStart64, uint32 lCount, void *lpBuffer, uint32 cbBuffer, uint32 *lBytesRead, uint32 *lSamplesRead)
{
	float pitch_Mult;
	int rawSize = 0;

	if (pixFormat == nsVDXPixmap::kPixFormat_RGB888) //mData._color
		pitch_Mult = 3;
	else if (pixFormat == nsVDXPixmap::kPixFormat_XRGB8888)
		pitch_Mult = 4;
	else if (pixFormat == nsVDXPixmap::kPixFormat_YUV422_YUYV)
		pitch_Mult = 2;
	else if (pixFormat == nsVDXPixmap::kPixFormat_YUV420_Planar)
		pitch_Mult = 1.5;

	if (mData.m_vFrameInfoDisplay.size() == 0) {
		// Dummy Video for Audio only file
		if (lStart64 < 0 || lStart64 >= mData.mHeader.mFrames) {
			if (lSamplesRead) *lSamplesRead = 0;
			if (lBytesRead) *lBytesRead = 0;
			return true;
		}

		if (lpBuffer == NULL || cbBuffer < mData.mHeader.mWidth * mData.mHeader.mHeight * pitch_Mult) {
			if (lSamplesRead) *lSamplesRead = 1;
			if (lBytesRead) *lBytesRead = mData.mHeader.mWidth * mData.mHeader.mHeight * pitch_Mult;
			return (lpBuffer == NULL);
		}

		*lBytesRead = mData.mHeader.mWidth * mData.mHeader.mHeight * pitch_Mult;
		*lSamplesRead = 1;

		return true;
	}

	if (lStart64 < 0 || lStart64 >= mData.mHeader.mFrames) {
		if (lSamplesRead) *lSamplesRead = 0;
		if (lBytesRead) *lBytesRead = 0;
		return true;
	}

	if (lpBuffer == NULL || cbBuffer < mData.mHeader.mWidth * mData.mHeader.mHeight * pitch_Mult) {
		if (lSamplesRead) *lSamplesRead = 1;
		if (lBytesRead) *lBytesRead = mData.mHeader.mWidth * mData.mHeader.mHeight * pitch_Mult;
		return (lpBuffer == NULL);
	}

	*lBytesRead = mData.mHeader.mWidth * mData.mHeader.mHeight * pitch_Mult;
	*lSamplesRead = 1;

	/*if (mPrevFrame != lStart64 + lCount - 1 || mPrevFrame < 0)*/ {
		int dst_pitch = mData.mHeader.mWidth * pitch_Mult;

		if (mData.m_iMode == QTMOVIE_MODE_VFW) {
			if (mData._color > 1) {
				if (pixFormat == nsVDXPixmap::kPixFormat_XRGB8888)
					mData.m_QTMovie->SetVFWOutput(k32BGRAPixelFormat);
				else if (pixFormat == nsVDXPixmap::kPixFormat_YUV422_YUYV)
					mData.m_QTMovie->SetVFWOutput(kYUVSPixelFormat);
				else if (pixFormat == nsVDXPixmap::kPixFormat_YUV420_Planar)
					mData.m_QTMovie->SetVFWOutput(kYUV420PixelFormat);
				else
					mData.m_QTMovie->SetVFWOutput(k24BGRPixelFormat);
			}
		}

		if (mData.m_iMode != QTMOVIE_MODE_VFW
				&& ((mData._color == 2 && pixFormat != nsVDXPixmap::kPixFormat_YUV422_YUYV)
					|| (mData._color == 3 && pixFormat != nsVDXPixmap::kPixFormat_YUV420_Planar)))
			*lBytesRead = mData.m_QTMovieRGB32->ReadVideoFrame(lStart64, (byte *)lpBuffer, dst_pitch, &rawSize);
		else
			*lBytesRead = mData.m_QTMovie->ReadVideoFrame(lStart64, (byte *)lpBuffer, dst_pitch, &rawSize);

		dst_pitch = *lBytesRead;
		*lBytesRead = rawSize;

		mPrevFrame = lStart64 + lCount - 1;
	}

	return true;
}

const void *VDVideoSourcePluginQuicktime::GetDirectFormat()
{
	return NULL;
}

int VDVideoSourcePluginQuicktime::GetDirectFormatLen()
{
	return 0;
}

IVDXStreamSource::ErrorMode VDVideoSourcePluginQuicktime::GetDecodeErrorMode()
{
	return IVDXStreamSource::kErrorModeReportAll;
}

void VDVideoSourcePluginQuicktime::SetDecodeErrorMode(IVDXStreamSource::ErrorMode mode)
{
}

bool VDVideoSourcePluginQuicktime::IsDecodeErrorModeSupported(IVDXStreamSource::ErrorMode mode)
{
	return mode == IVDXStreamSource::kErrorModeReportAll;
}

bool VDVideoSourcePluginQuicktime::IsVBR()
{
	return false;
}

sint64 VDVideoSourcePluginQuicktime::TimeToPositionVBR(sint64 us)
{
	return (sint64)(0.5 + us / 1000000.0 * (double)mData.mHeader.mFrameRateNumerator / (double)mData.mHeader.mFrameRateDenominator);
}

sint64 VDVideoSourcePluginQuicktime::PositionToTimeVBR(sint64 samples)
{//JHH
	return (sint64)(0.5 + samples * 1000000.0 * (double)mData.mHeader.mFrameRateDenominator / (double)mData.mHeader.mFrameRateNumerator);
}

void VDVideoSourcePluginQuicktime::GetVideoSourceInfo(VDXVideoSourceInfo& info)
{
	info.mFlags = 0;
	info.mWidth = mData.mHeader.mWidth;
	info.mHeight = mData.mHeader.mHeight;
	info.mDecoderModel = VDXVideoSourceInfo::kDecoderModelCustom;
	// kDecoderModelDefaultIP kDecoderModelCustom
}

bool VDVideoSourcePluginQuicktime::CreateVideoDecoderModel(IVDXVideoDecoderModel **ppModel)
{
	VDVideoDecoderModelPluginQuicktime *p = new VDVideoDecoderModelPluginQuicktime(mData);
	if (!p)
		return false;

	p->AddRef();
	*ppModel = p;
	return true;
}

bool VDVideoSourcePluginQuicktime::CreateVideoDecoder(IVDXVideoDecoder **ppDecoder)
{
	VDVideoDecoderPluginQuicktime *p = new VDVideoDecoderPluginQuicktime(mData);
	if (!p)
		return false;
	p->AddRef();
	*ppDecoder = p;
	return true;
}

void VDVideoSourcePluginQuicktime::GetSampleInfo(sint64 sample_num, VDXVideoFrameInfo& frameInfo)
{
	frameInfo.mBytePosition = -1;

	//frameInfo.mFrameType = kVDXVFT_Independent;
	//frameInfo.mTypeChar = 'I';
	//return;

	if (sample_num >= mData.m_vFrameInfoDisplay.size())
		return;

	switch(mData.m_vFrameInfoDisplay[(uint32)sample_num].mFrameType) {
		case 0:
			frameInfo.mFrameType = kVDXVFT_Independent;
			frameInfo.mTypeChar = 'I';
			break;
		case 1:
			frameInfo.mFrameType = kVDXVFT_Predicted;
			frameInfo.mTypeChar = 'P';
			break;
		case 2:
			frameInfo.mFrameType = kVDXVFT_Bidirectional;
			frameInfo.mTypeChar = 'B';
			break;
	}
}

bool VDVideoSourcePluginQuicktime::IsKey(sint64 sample)
{
	if (sample >= mData.m_vFrameInfoDisplay.size())
		return true;

	return (mData.m_vFrameInfoDisplay[(uint32)sample].mFrameType == 0);
}

sint64 VDVideoSourcePluginQuicktime::GetFrameNumberForSample(sint64 sample_num)
{
	return sample_num;
/*
	if (mData.m_vFrameInfoDisplay[(uint32)sample_num].mFrameType == 2)
		return sample_num - 1;
	else {
		uint32 n = (uint32)mData.m_vFrameInfoDisplay.size();

		for(uint32 i = (uint32)sample_num + 1; i<n && mData.m_vFrameInfoDisplay[i].mFrameType == 2; ++i)
			++sample_num;

		return sample_num;
	}
*/
}

sint64 VDVideoSourcePluginQuicktime::GetSampleNumberForFrame(sint64 display_num)
{
	return display_num;

/*
	uint32 n = (uint32)mData.m_vFrameInfoDisplay.size();
	uint32 i = (uint32)display_num + 1;

	if (i < n && mData.m_vFrameInfoDisplay[i].mFrameType == 2) {
		return i;
	} else {
		do {
			--i;

			if (mData.m_vFrameInfoDisplay[i].mFrameType != 2)
				break;

			--display_num;
		} while(i > 0);

		return display_num;
	}
*/
}

sint64 VDVideoSourcePluginQuicktime::GetRealFrame(sint64 display_num)
{
	return display_num;
}

sint64 VDVideoSourcePluginQuicktime::GetSampleBytePosition(sint64 sample_num)
{
	return -1;
}

///////////////////////////////////////////////////////////////////////////////

class VDAudioSourcePluginQuicktime : public vdxunknown<IVDXStreamSource>, public IVDXAudioSource {
public:
	VDAudioSourcePluginQuicktime(QuicktimeData& data);
	~VDAudioSourcePluginQuicktime();

	int VDXAPIENTRY AddRef();
	int VDXAPIENTRY Release();
	void *VDXAPIENTRY AsInterface(uint32 iid);

	void		VDXAPIENTRY GetStreamSourceInfo(VDXStreamSourceInfo&);
	bool		VDXAPIENTRY Read(sint64 lStart, uint32 lCount, void *lpBuffer, uint32 cbBuffer, uint32 *lBytesRead, uint32 *lSamplesRead);

	const void *VDXAPIENTRY GetDirectFormat();
	int			VDXAPIENTRY GetDirectFormatLen();

	ErrorMode VDXAPIENTRY GetDecodeErrorMode();
	void VDXAPIENTRY SetDecodeErrorMode(ErrorMode mode);
	bool VDXAPIENTRY IsDecodeErrorModeSupported(ErrorMode mode);

	bool VDXAPIENTRY IsVBR();
	sint64 VDXAPIENTRY TimeToPositionVBR(sint64 us);
	sint64 VDXAPIENTRY PositionToTimeVBR(sint64 samples);

	void VDXAPIENTRY GetAudioSourceInfo(VDXAudioSourceInfo& info);

protected:
	VDXWAVEFORMATEX mRawFormat;

	QuicktimeData&	mData;

	uint64		mLength;
};

VDAudioSourcePluginQuicktime::VDAudioSourcePluginQuicktime(QuicktimeData& data)
	: mData(data)
{
	mRawFormat.mFormatTag		= mData.mHeader.mFormatTag;
	mRawFormat.mChannels		= mData.mHeader.mChannels;
	mRawFormat.mSamplesPerSec	= mData.mHeader.mSamplesPerSec;
	mRawFormat.mAvgBytesPerSec	= mData.mHeader.mAvgBytesPerSec;
	mRawFormat.mBlockAlign		= mData.mHeader.mChannels * mData.mHeader.mBitsPerSample / 8;
	mRawFormat.mBitsPerSample	= mData.mHeader.mBitsPerSample;
	mRawFormat.mExtraSize		= 0;

	//mLength = (uint64)(0.5 + mData.mHeader.mFrames * mData.mHeader.mSamplesPerSec * (double)data.mHeader.mFrameRateDenominator / (double)data.mHeader.mFrameRateNumerator);
	mLength = mData.mHeader.mSampleCount;

	/*mRawFormat.mFormatTag		= VDXWAVEFORMATEX::kFormatPCM;
	mRawFormat.mChannels		= 1;
	mRawFormat.mSamplesPerSec	= 44100;
	mRawFormat.mAvgBytesPerSec	= 44100*2;
	mRawFormat.mBlockAlign		= 2;
	mRawFormat.mBitsPerSample	= 16;
	mRawFormat.mExtraSize		= 0;

	mLength = (uint64)(0.5 + data.mHeader.mFrames * 44100.0 * (double)data.mHeader.mFrameRateDenominator / (double)data.mHeader.mFrameRateNumerator);*/
}

VDAudioSourcePluginQuicktime::~VDAudioSourcePluginQuicktime()
{
}

int VDAudioSourcePluginQuicktime::AddRef()
{
	return vdxunknown<IVDXStreamSource>::AddRef();
}

int VDAudioSourcePluginQuicktime::Release()
{
	return vdxunknown<IVDXStreamSource>::Release();
}

void *VDXAPIENTRY VDAudioSourcePluginQuicktime::AsInterface(uint32 iid)
{
	if (iid == IVDXVideoSource::kIID)
		return static_cast<IVDXAudioSource *>(this);

	return vdxunknown<IVDXStreamSource>::AsInterface(iid);
}

void VDXAPIENTRY VDAudioSourcePluginQuicktime::GetStreamSourceInfo(VDXStreamSourceInfo& srcInfo)
{
	srcInfo.mSampleRate = mData.mHeader.mSampleRate;
	srcInfo.mSampleCount = mLength;

	/*srcInfo.mSampleRate.mNumerator = 44100;
	srcInfo.mSampleRate.mDenominator = 1;
	srcInfo.mSampleCount = mLength;*/
}

bool VDAudioSourcePluginQuicktime::Read(sint64 lStart64, uint32 lCount, void *lpBuffer, uint32 cbBuffer, uint32 *lBytesRead, uint32 *lSamplesRead)
{
	uint32 m_iBytesConverted = 0;
	uint32 start = (uint32)lStart64;

	if (lStart64 >= mData.mHeader.mSampleCount)
		return false;

	if (lStart64 + lCount > mData.mHeader.mSampleCount)
		lCount = mData.mHeader.mSampleCount - lStart64 - 1;

	if (! lCount) {
		*lSamplesRead = 1;
		*lBytesRead = *lSamplesRead * mData.mHeader.mBlockAlign;
		return false;
	}

	*lSamplesRead = lCount;
	*lBytesRead = *lSamplesRead * mData.mHeader.mBlockAlign;

	m_iBytesConverted = *lBytesRead;
	m_iBytesConverted = *lSamplesRead;

	if (! lpBuffer)
		return true;

	*lSamplesRead = mData.m_QTMovie->ReadAudioFrame((byte *)lpBuffer, lStart64, lCount, lBytesRead);
	return true;
}

const void *VDAudioSourcePluginQuicktime::GetDirectFormat()
{
	return &mRawFormat;
}

int VDAudioSourcePluginQuicktime::GetDirectFormatLen()
{
	return sizeof(mRawFormat);
}

IVDXStreamSource::ErrorMode VDAudioSourcePluginQuicktime::GetDecodeErrorMode()
{
	return IVDXStreamSource::kErrorModeReportAll;
}

void VDAudioSourcePluginQuicktime::SetDecodeErrorMode(IVDXStreamSource::ErrorMode mode)
{
}

bool VDAudioSourcePluginQuicktime::IsDecodeErrorModeSupported(IVDXStreamSource::ErrorMode mode)
{
	return mode == IVDXStreamSource::kErrorModeReportAll;
}

bool VDAudioSourcePluginQuicktime::IsVBR()
{
	return false;
}

sint64 VDAudioSourcePluginQuicktime::TimeToPositionVBR(sint64 us)
{
	return (sint64)(0.5 + us / 1000000.0 * (double)mData.mHeader.mFrameRateNumerator / (double)mData.mHeader.mFrameRateDenominator);
}

sint64 VDAudioSourcePluginQuicktime::PositionToTimeVBR(sint64 samples)
{
	return (sint64)(0.5 + samples * 1000000.0 * (double)mData.mHeader.mFrameRateDenominator / (double)mData.mHeader.mFrameRateNumerator);
}

void VDAudioSourcePluginQuicktime::GetAudioSourceInfo(VDXAudioSourceInfo& info)
{
	info.mFlags = 0;
}

///////////////////////////////////////////////////////////////////////////////

class VDInputFileQuicktimeOptions : public vdxunknown<IVDXInputOptions> {
public:
	struct InputFileQuicktimeOpts {
		int len;
		int _vtrack, _atrack, _color, _audio; // _dither _raw
		int m_iMode, quality;
		float m_fGamma;
		char strVFW[5];
	} opts;
	
	VDInputFileQuicktimeOptions();
	~VDInputFileQuicktimeOptions();

	//bool read(const char *buf);
	uint32 VDXAPIENTRY Write(void *buf, uint32 buflen);

	static INT_PTR APIENTRY OptionsDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam);
};

VDInputFileQuicktimeOptions::VDInputFileQuicktimeOptions()
{
	opts.len = sizeof(opts);
	opts._color = -1;
	opts.m_iMode = -1;
	opts._audio = 1;
	opts.quality = 100;
	opts._vtrack = 1;
	opts._atrack = 1;
	opts.m_fGamma = 2.5;
	strcpy(opts.strVFW, "");

	/*wchar_t *path = new wchar_t[MAX_PATH];
	wchar_t *drive = new wchar_t[MAX_DRIVE];
	wchar_t *dir = new wchar_t[MAX_DIR];
	wchar_t *fname = new wchar_t[MAX_FNAME];
	wchar_t *ext = new wchar_t[MAX_EXT];*/

	char *iniFile = new char[MAX_PATH];
	char *path = new char[MAX_PATH];
	char *drive = new char[MAX_DRIVE];
	char *dir = new char[MAX_DIR];
	char *fname = new char[MAX_FNAME];
	char *ext = new char[MAX_EXT];

	// Read from ini file
	if (GetModuleFileNameA(NULL, path, MAX_PATH)) {
		_splitpath(path, drive, dir, fname, ext);

		sprintf(iniFile, "%s%squicktime.ini", drive, dir);

		opts._color = GetPrivateProfileInt("Settings", "color", -1, iniFile);
		opts.m_iMode = GetPrivateProfileInt("Settings", "Mode", -1, iniFile);
		opts._audio = GetPrivateProfileInt("Settings", "audio", 1, iniFile);
		opts.quality = GetPrivateProfileInt("Settings", "quality", 100, iniFile);
		opts._vtrack = GetPrivateProfileInt("Settings", "vtrack", 1, iniFile);
		opts._atrack = GetPrivateProfileInt("Settings", "atrack", 1, iniFile);

		GetPrivateProfileString("Settings", "gamma", "2.5", opts.strVFW, 4, iniFile);
			opts.m_fGamma = atof(opts.strVFW);
		GetPrivateProfileString("Settings", "fourcc", "", opts.strVFW, 5, iniFile);
	}

	delete [] iniFile;
	delete [] path;
	delete [] drive;
	delete [] dir;
	delete [] fname;
	delete [] ext;
}

VDInputFileQuicktimeOptions::~VDInputFileQuicktimeOptions()
{
}

uint32 VDInputFileQuicktimeOptions::Write(void *buf, uint32 buflen)
{
	if (buf != NULL && buflen >= opts.len) {
		memcpy(buf, &opts, opts.len);
	}
	return opts.len;
}

INT_PTR APIENTRY VDInputFileQuicktimeOptions::OptionsDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
 	VDInputFileQuicktimeOptions *thisPtr;
		//= (VDInputFileQuicktimeOptions *)GetWindowLongPtr(hDlg, DWLP_USER);
	HWND hwndItem;
	char str[5];

	char *iniFile;
	char *path;
	char *drive;
	char *dir;
	char *fname;
	char *ext;

	switch (message)
	{
		case WM_INITDIALOG:
				SetWindowLongPtr(hDlg, DWLP_USER, lParam);
				thisPtr = (VDInputFileQuicktimeOptions *)lParam;

				if (! thisPtr)
					return FALSE;
				
				hwndItem = GetDlgItem(hDlg, IDC_QUICKTIME_COLOR);
					SendMessage(hwndItem, CB_RESETCONTENT, 0, 0);
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)"-1 : Auto");
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)" 0 : RGB24");
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)" 1 : RGB32");
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)" 2 : YUY2");
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)" 3 : YV12");
					SendMessage(hwndItem, CB_SETCURSEL, thisPtr->opts._color + 1, 0);
				hwndItem = GetDlgItem(hDlg, IDC_QUICKTIME_MODE);
					SendMessage(hwndItem, CB_RESETCONTENT, 0, 0);
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)"-1 : Auto");
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)" 0 : MoviesTask");
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)" 1 : Does not work");
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)" 2 : Decomp Sequence");
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)" 3 : ICM Decomp");
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)" 4 : VFW");
					SendMessage(hwndItem, CB_SETCURSEL, 0, 0);
					SendMessage(hwndItem, CB_SETCURSEL, thisPtr->opts.m_iMode + 1, 0);
				hwndItem = GetDlgItem(hDlg, IDC_QUICKTIME_AUDIO);
					SendMessage(hwndItem, CB_RESETCONTENT, 0, 0);
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)" 0 : Disabled");
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)" 1 : Enabled");
					SendMessage(hwndItem, CB_SETCURSEL, 0, 0);
					SendMessage(hwndItem, CB_SETCURSEL, thisPtr->opts._audio, 0);

				SetDlgItemInt(hDlg, IDC_QUICKTIME_QUALITY, thisPtr->opts.quality, FALSE);
				SetDlgItemInt(hDlg, IDC_QUICKTIME_VTRACK, thisPtr->opts._vtrack, FALSE);
				SetDlgItemInt(hDlg, IDC_QUICKTIME_ATRACK, thisPtr->opts._atrack, FALSE);
				sprintf_s(str, 5, "%0.2f", thisPtr->opts.m_fGamma)
					SetDlgItemText(hDlg, IDC_QUICKTIME_GAMMA, str);
				SetDlgItemText(hDlg, IDC_QUICKTIME_FOURCC, thisPtr->opts.strVFW);
				return TRUE;
		case WM_COMMAND:
			thisPtr = (VDInputFileQuicktimeOptions *)GetWindowLongPtr(hDlg, DWLP_USER);
			switch(LOWORD(wParam)) {
			case IDOK:
				hwndItem = GetDlgItem(hDlg, IDC_QUICKTIME_COLOR);
					thisPtr->opts._color = SendMessage(hwndItem, CB_GETCURSEL, 0, 0) - 1;
				hwndItem = GetDlgItem(hDlg, IDC_QUICKTIME_MODE);
					thisPtr->opts.m_iMode = SendMessage(hwndItem, CB_GETCURSEL, 0, 0) - 1;
				hwndItem = GetDlgItem(hDlg, IDC_QUICKTIME_AUDIO);
					thisPtr->opts._audio = SendMessage(hwndItem, CB_GETCURSEL, 0, 0);

				thisPtr->opts._vtrack = GetDlgItemInt(hDlg, IDC_QUICKTIME_VTRACK, NULL, FALSE);
				thisPtr->opts._atrack = GetDlgItemInt(hDlg, IDC_QUICKTIME_ATRACK, NULL, FALSE);
				thisPtr->opts.quality = GetDlgItemInt(hDlg, IDC_QUICKTIME_QUALITY, NULL, FALSE);

				GetDlgItemText(hDlg, IDC_QUICKTIME_GAMMA, str, 5);
				thisPtr->opts.m_fGamma = atof(str);
				GetDlgItemText(hDlg, IDC_QUICKTIME_FOURCC, thisPtr->opts.strVFW, 5);
				EndDialog(hDlg, TRUE);
			case IDCANCEL:
				EndDialog(hDlg, TRUE);
				return FALSE;
			case IDC_QUICKTIME_DEFAULT:
				hwndItem = GetDlgItem(hDlg, IDC_QUICKTIME_COLOR);
					SendMessage(hwndItem, CB_SETCURSEL, 0, 0);
				hwndItem = GetDlgItem(hDlg, IDC_QUICKTIME_MODE);
					SendMessage(hwndItem, CB_SETCURSEL, 0, 0);
				hwndItem = GetDlgItem(hDlg, IDC_QUICKTIME_AUDIO);
					SendMessage(hwndItem, CB_SETCURSEL, 1, 0);

				SetDlgItemInt(hDlg, IDC_QUICKTIME_QUALITY, 100, FALSE);
				SetDlgItemInt(hDlg, IDC_QUICKTIME_VTRACK, 1, FALSE);
				SetDlgItemInt(hDlg, IDC_QUICKTIME_ATRACK, 1, FALSE);
				SetDlgItemText(hDlg, IDC_QUICKTIME_GAMMA, "2.5");
				SetDlgItemText(hDlg, IDC_QUICKTIME_FOURCC, "");
				return TRUE;
			case IDC_QUICKTIME_SAVE:
				hwndItem = GetDlgItem(hDlg, IDC_QUICKTIME_COLOR);
					thisPtr->opts._color = SendMessage(hwndItem, CB_GETCURSEL, 0, 0) - 1;
				hwndItem = GetDlgItem(hDlg, IDC_QUICKTIME_MODE);
					thisPtr->opts.m_iMode = SendMessage(hwndItem, CB_GETCURSEL, 0, 0) - 1;
				hwndItem = GetDlgItem(hDlg, IDC_QUICKTIME_AUDIO);
					thisPtr->opts._audio = SendMessage(hwndItem, CB_GETCURSEL, 0, 0);

				thisPtr->opts._vtrack = GetDlgItemInt(hDlg, IDC_QUICKTIME_VTRACK, NULL, FALSE);
				thisPtr->opts._atrack = GetDlgItemInt(hDlg, IDC_QUICKTIME_ATRACK, NULL, FALSE);
				thisPtr->opts.quality = GetDlgItemInt(hDlg, IDC_QUICKTIME_QUALITY, NULL, FALSE);

				GetDlgItemText(hDlg, IDC_QUICKTIME_GAMMA, str, 5);
				thisPtr->opts.m_fGamma = atof(str);
				GetDlgItemText(hDlg, IDC_QUICKTIME_FOURCC, thisPtr->opts.strVFW, 5);

				iniFile = new char[MAX_PATH];
				path = new char[MAX_PATH];
				drive = new char[MAX_DRIVE];
				dir = new char[MAX_DIR];
				fname = new char[MAX_FNAME];
				ext = new char[MAX_EXT];

				char val[10];

				// Write to ini file
				if (GetModuleFileNameA(NULL, path, MAX_PATH)) {
					_splitpath(path, drive, dir, fname, ext);

					sprintf(iniFile, "%s%squicktime.ini", drive, dir);

					sprintf(val, "%d", thisPtr->opts._color);
						WritePrivateProfileString("Settings", "color", val, iniFile);
					sprintf(val, "%d", thisPtr->opts.m_iMode);
						WritePrivateProfileString("Settings", "Mode", val, iniFile);
					sprintf(val, "%d", thisPtr->opts._audio);
						WritePrivateProfileString("Settings", "audio", val, iniFile);
					sprintf(val, "%d", thisPtr->opts.quality);
						WritePrivateProfileString("Settings", "quality", val, iniFile);
					sprintf(val, "%d", thisPtr->opts._vtrack);
						WritePrivateProfileString("Settings", "vtrack", val, iniFile);
					sprintf(val, "%d", thisPtr->opts._atrack);
						WritePrivateProfileString("Settings", "atrack", val, iniFile);
					sprintf(val, "%0.2f", thisPtr->opts.m_fGamma);
						WritePrivateProfileString("Settings", "gamma", val, iniFile);
					WritePrivateProfileString("Settings", "fourcc", thisPtr->opts.strVFW, iniFile);
				} else {
					MessageBox(hDlg, "Could not save to quicktime.ini", "Error", MB_OK);
				}

				delete [] iniFile;
				delete [] path;
				delete [] drive;
				delete [] dir;
				delete [] fname;
				delete [] ext;

				return TRUE;
			default:
				break;
			}
			break;
	}
	return FALSE;
}

///////////////////////////////////////////////////////////////////////////////

class VDInputFilePluginQuicktime : public vdxunknown<IVDXInputFile> {
public:
	VDInputFilePluginQuicktime(const VDInputDriverContext& context);
	~VDInputFilePluginQuicktime();

	void VDXAPIENTRY Init(const wchar_t *szFile, IVDXInputOptions *opts);
	bool VDXAPIENTRY Append(const wchar_t *szFile);

	bool VDXAPIENTRY PromptForOptions(VDXHWND hwndParent, IVDXInputOptions **);
	bool VDXAPIENTRY CreateOptions(const void *buf, uint32 len, IVDXInputOptions **);
	void VDXAPIENTRY DisplayInfo(VDXHWND hwndParent);

	bool VDXAPIENTRY GetVideoSource(int index, IVDXVideoSource **);
	bool VDXAPIENTRY GetAudioSource(int index, IVDXAudioSource **);

	static INT_PTR APIENTRY InfoDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam);

protected:
	QuicktimeData	mData;
	const VDInputDriverContext& mContext;
};

VDInputFilePluginQuicktime::VDInputFilePluginQuicktime(const VDInputDriverContext& context)
	: mContext(context)
{
	mData.m_QTMovie = NULL;
	mData.m_QTMovieRGB32 = NULL;

	mData._vtrack = 1;
	mData._atrack = 1;
	mData._color = -1;
	mData._raw = 0;
	mData._audio = 1;
	mData.m_iMode = QTMOVIE_MODE_AUTO;
	mData.quality = 100;
	mData._dither = 1;
	mData.m_fGamma = 2.5;
}

VDInputFilePluginQuicktime::~VDInputFilePluginQuicktime()
{
	if (mData.m_QTMovie) {
		delete mData.m_QTMovie;
		mData.m_QTMovie = NULL;
	}
	if (mData.m_QTMovieRGB32) {
		delete mData.m_QTMovieRGB32;
		mData.m_QTMovieRGB32 = NULL;
	}
}

void VDInputFilePluginQuicktime::Init(const wchar_t *szFile, IVDXInputOptions *opts)
{
	char abuf[1024];
	int i;
	bool bAutoMode = false;
	bool bAutoColor = true;

	wcstombs(abuf, szFile, 1024);
	abuf[1023] = 0;

	mData.m_QTMovie = new CQTMovieDec;

	if (! mData.m_QTMovie->ISQTInitialized()) {
		//sprintf(strStatus, "\n%s\n", mData.m_QTMovie->GetErrorMessage());
		mContext.mpCallbacks->SetError("Unable to open file: %ls\n%s",
			szFile, mData.m_QTMovie->GetErrorMessage());
		//delete mData.m_QTMovie;
		return;
	}

	const VDInputFileQuicktimeOptions *pOptions = (VDInputFileQuicktimeOptions *)opts;

	if (pOptions != NULL) {
		mData._color = pOptions->opts._color;
		mData.m_iMode = pOptions->opts.m_iMode;
		mData._audio = pOptions->opts._audio;
		mData.quality = pOptions->opts.quality;
		mData._vtrack = pOptions->opts._vtrack;
		mData._atrack = pOptions->opts._atrack;
		mData.m_fGamma = pOptions->opts.m_fGamma;
		strcpy(mData.strVFW, pOptions->opts.strVFW);
	} else {
		/*wchar_t *path = new wchar_t[MAX_PATH];
		wchar_t *drive = new wchar_t[MAX_DRIVE];
		wchar_t *dir = new wchar_t[MAX_DIR];
		wchar_t *fname = new wchar_t[MAX_FNAME];
		wchar_t *ext = new wchar_t[MAX_EXT];*/

		char *iniFile = new char[MAX_PATH];
		char *path = new char[MAX_PATH];
		char *drive = new char[MAX_DRIVE];
		char *dir = new char[MAX_DIR];
		char *fname = new char[MAX_FNAME];
		char *ext = new char[MAX_EXT];

		// Read from ini file
		if (GetModuleFileNameA(NULL, path, MAX_PATH)) {
			_splitpath(path, drive, dir, fname, ext);

			sprintf(iniFile, "%s%squicktime.ini", drive, dir);

			mData._color = GetPrivateProfileInt("Settings", "color", -1, iniFile);
			mData.m_iMode = GetPrivateProfileInt("Settings", "Mode", -1, iniFile);
			mData._audio = GetPrivateProfileInt("Settings", "audio", 1, iniFile);
			mData.quality = GetPrivateProfileInt("Settings", "quality", 100, iniFile);
			mData._vtrack = GetPrivateProfileInt("Settings", "vtrack", 1, iniFile);
			mData._atrack = GetPrivateProfileInt("Settings", "atrack", 1, iniFile);

			GetPrivateProfileString("Settings", "gamma", "2.5", mData.strVFW, 4, iniFile);
				mData.m_fGamma = atof(mData.strVFW);
			GetPrivateProfileString("Settings", "fourcc", "", mData.strVFW, 5, iniFile);
		}

		delete [] iniFile;
		delete [] path;
		delete [] drive;
		delete [] dir;
		delete [] fname;
		delete [] ext;
	}

	mData.mCodec.quality =
		codecQuality[mData.quality > 100 ? 5 : (mData.quality < 0
		? 0 : (int)(mData.quality/20))];

	memset(mData.mCodec.CodecName, 0, sizeof(mData.mCodec.CodecName));
	memset(mData.mCodec.strFourCC, 0, sizeof(mData.mCodec.strFourCC));

	if (mData.m_QTMovie->OpenMovie(abuf, mData.mCodec, mData._vtrack, mData._atrack) == E_FAIL) {
		//sprintf(strStatus, "\n%s\n", mData.m_QTMovie->GetErrorMessage());
		mContext.mpCallbacks->SetError("Unable to open movie: %ls\n%s",
			szFile, mData.m_QTMovie->GetErrorMessage());
		//delete mData.m_QTMovie;
		return;
	}

	mData.mCodec = mData.m_QTMovie->QTGetCodecInfo();
	OSType m_pixelFormatArray[4];

SET_AUTO_MODE:
	if (mData.m_iMode == QTMOVIE_MODE_AUTO) {
		bAutoMode = true;
		if (
				   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('2Vuy') // Blackmagic 8 Bit (2Vuy)
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('2vuy') // Blackmagic 8 Bit
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AV1x') // Avid 1:1x
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVUI') // Avid Meridien Uncompressed
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVup') // Avid Packed Codec
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('r210') // Blackmagic RGB 10 Bit
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('v210') // Blackmagic 10 Bit
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('yuv2') // Component Video
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('raw ') // None
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('NO16') // None16
		) // Eventually should turn on Avid Codecs here, but need to figure out frame offset
			mData.m_iMode = QTMOVIE_MODE_VFW; // QTMOVIE_MODE_RAW
		else if (
				   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVdv') // Avid DV
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dvc ') // DV/DVCPRO - NTSC
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dv25') // DV25
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dvcp') // DV - PAL
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dvpp') // DVCPRO - PAL
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dv50') // DV50
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dv5n') // DV50 - NTSC
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dv5p') // DV50 - PAL
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVmp') // AVID Mpeg-IMX
		) // need to figure out frame offset for Mpeg-IMX
			mData.m_iMode = QTMOVIE_MODE_VFW;
		else if (
				   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('jpeg') // Photo - JPEG
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('mjp2') // JPEG 2000
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('mjpa') // Motion JPEG A
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVDJ') // Avid Meridien Compressed
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVRn') // Avid
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('mjpb') // Motion JPEG B
		) // Eventually should turn on Avid Codecs here, but need to figure out frame offset
			mData.m_iMode = QTMOVIE_MODE_VFW;
		else if (
				   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('mp4v') // MPEG-4 Video
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('SVQ1') // Sorenson Video
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('SVQ3') // Sorenson Video 3
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('h261') // H.261
				//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('h263') // H.263
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('cvid') // Cinepak
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('IV41') // Intel Indeo Video 4.4
		)
			mData.m_iMode = QTMOVIE_MODE_VFW;
		else if (
				    mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('avc1') // H.264
		)
			mData.m_iMode = QTMOVIE_MODE_VFW;
		else if (mData.m_QTMovie->IsIFrameOnly())
			mData.m_iMode = QTMOVIE_MODE_DESQNC;
		else //if (! mData.m_QTMovie->IsIFrameOnly())
			mData.m_iMode = QTMOVIE_MODE_ICM;
		/*else if (
				   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('WRLE') // BMP
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('png ') // PNG
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('rle ') // Animation
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('rpza') // Video
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('smc ') // Graphics
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('tga ') // TGA
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('tiff') // TIFF
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('8BPS') // Planar RGB
		)
			mData.m_iMode = QTMOVIE_MODE_DESQNC;
		else if (
				   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVd1') // Avid DV100 Codec
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVdn') // Avid DNxHD Codec
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('HD10') // Accom WSD/HD Video
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('HDK1') // Accom WSD/HD Key
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('ImJG') // Sphere Video
		)
			mData.m_iMode = QTMOVIE_MODE_DESQNC;
		else
			mData.m_iMode = QTMOVIE_MODE_DESQNC;*/
	} else if (mData.m_iMode == QTMOVIE_MODE_ICM && (
				   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVup')
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVdv')
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVRn')
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVDJ')
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('2Vuy')
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('r210')
				|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('2vuy'))) {
		// Use m_iMode = 2 for these Codecs
		// ICMDecompression, m_iMode = 3, does not work
		mData.m_iMode = QTMOVIE_MODE_DESQNC;
	}

	if (mData._color == QTMOVIE_MODE_AUTO) {
		bAutoColor = true;
		if (mData.m_iMode == QTMOVIE_MODE_VFW) {
			if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('2Vuy') // Blackmagic 8 Bit (2Vuy)
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('2vuy') // Blackmagic 8 Bit
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AV1x') // Avid 1:1x
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVUI') // Avid Meridien Uncompressed
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVup') // Avid Packed Codec
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('v210') // Blackmagic 10 Bit
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('yuv2') // Component Video
			)
				mData._color = 2; // YUY2
			else if (
					    mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('r210') // Blackmagic RGB 10 Bit
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('raw ') // None
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('NO16') // None16
			)
				mData._color = 1; // RGB32
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVdv') // Avid DV
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dvc ') // DV/DVCPRO - NTSC
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dv25') // DV25
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dvcp') // DV - PAL
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dvpp') // DVCPRO - PAL
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dv50') // DV50
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dv5n') // DV50 - NTSC
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dv5p') // DV50 - PAL
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVmp') // AVID Mpeg-IMX
			)
				mData._color = 2; // YUY2
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVDJ') // Avid Meridien Compressed
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVRn') // Avid
			)
				mData._color = 2; // YUY2
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('jpeg') // Photo - JPEG
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('mjp2') // JPEG 2000
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('mjpa') // Motion JPEG A
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('mjpb') // Motion JPEG B
			)
				mData._color = 0; // RGB24
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('mp4v') // MPEG-4 Video
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('SVQ1') // Sorenson Video
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('SVQ3') // Sorenson Video 3
					//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('h261') // H.261
					//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('h263') // H.263
					//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('cvid') // Cinepak
					//|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('IV41') // Intel Indeo Video 4.4
			)
				mData._color = 2; // YUY2
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('avc1') // H.264
			)
				mData._color = 2; // YUY2
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('WRLE') // BMP
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('png ') // PNG
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('rle ') // Animation
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('rpza') // Video
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('smc ') // Graphics
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('tga ') // TGA
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('tiff') // TIFF
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('8BPS') // Planar RGB
			)
				mData._color = 1; // RGB32
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVd1') // Avid DV100 Codec
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVdn') // Avid DNxHD Codec
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('HD10') // Accom WSD/HD Video
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('HDK1') // Accom WSD/HD Key
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('ImJG') // Sphere Video
			)
				mData._color = 1; // RGB32
			else
				mData._color = 1; // RGB32
		} else if (mData.m_iMode == QTMOVIE_MODE_RAW) {
			if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('2Vuy') // Blackmagic 8 Bit (2Vuy)
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('2vuy') // Blackmagic 8 Bit
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AV1x') // Avid 1:1x
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVUI') // Avid Meridien Uncompressed
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVup') // Avid Packed Codec
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('v210') // Blackmagic 10 Bit
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('yuv2') // Component Video
			)
				mData._color = 2; // YUY2
			else if (
					    mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('r210') // Blackmagic RGB 10 Bit
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('raw ') // None
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('NO16') // None16
			)
				mData._color = 1; // RGB32
			else
				mData._color = 1; // RGB32
		} else {
			if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('2Vuy') // Blackmagic 8 Bit (2Vuy)
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('2vuy') // Blackmagic 8 Bit
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AV1x') // Avid 1:1x
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVUI') // Avid Meridien Uncompressed
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVup') // Avid Packed Codec
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('v210') // Blackmagic 10 Bit
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('yuv2') // Component Video
			)
				mData._color = 1; // RGB32
			else if (
					    mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('r210') // Blackmagic RGB 10 Bit
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('raw ') // None
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('NO16') // None16
			)
				mData._color = 1; // RGB32
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVdv') // Avid DV
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dvc ') // DV/DVCPRO - NTSC
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dv25') // DV25
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dvcp') // DV - PAL
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dvpp') // DVCPRO - PAL
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dv50') // DV50
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dv5n') // DV50 - NTSC
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('dv5p') // DV50 - PAL
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVmp') // AVID Mpeg-IMX
			)
				mData._color = 1; // RGB32
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVDJ') // Avid Meridien Compressed
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVRn') // Avid
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('jpeg') // Photo - JPEG
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('mjp2') // JPEG 2000
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('mjpa') // Motion JPEG A
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('mjpb') // Motion JPEG B
			)
				mData._color = 1; // RGB32
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('mp4v') // MPEG-4 Video
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('SVQ1') // Sorenson Video
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('SVQ3') // Sorenson Video 3
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('h261') // H.261
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('h263') // H.263
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('cvid') // Cinepak
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('IV41') // Intel Indeo Video 4.4
			)
				mData._color = 1; // RGB32
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('avc1') // H.264
			)
				mData._color = 1; // RGB32
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('WRLE') // BMP
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('png ') // PNG
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('rle ') // Animation
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('rpza') // Video
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('smc ') // Graphics
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('tga ') // TGA
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('tiff') // TIFF
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('8BPS') // Planar RGB
			)
				mData._color = 1; // RGB32
			else if (
					   mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVd1') // Avid DV100 Codec
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('AVdn') // Avid DNxHD Codec
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('HD10') // Accom WSD/HD Video
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('HDK1') // Accom WSD/HD Key
					|| mData.mCodec.m_iFourCC == FOUR_CHAR_CODE('ImJG') // Sphere Video
			)
				mData._color = 1; // RGB32
			else
				mData._color = 1; // RGB32
		}
	}

	if (mData._color == 3 && mData.m_iMode != QTMOVIE_MODE_VFW) {
		int retVal = MessageBox(GetActiveWindow(),
			"YV12 is only suported for VFW, mode=4.\n\nWould you like to try again using RGB32?",
			"Not Supported", MB_YESNO);

		if (retVal == 6) {
			if (bAutoMode)
				mData.m_iMode = -1;
			mData._color = 1;
			goto SET_AUTO_MODE;
		}
	}

	if (mData._raw != 0) {
		if (mData._raw == QTMOVIE_RAW_MODE_RGB) {
			m_pixelFormatArray[0] = k24BGRPixelFormat;
			m_pixelFormatArray[1] = k24BGRPixelFormat;
			m_pixelFormatArray[2] = k24BGRPixelFormat;
			m_pixelFormatArray[3] = k24BGRPixelFormat;
		} else if (mData._raw == QTMOVIE_RAW_MODE_ARGB) {
			m_pixelFormatArray[0] = k32BGRAPixelFormat;
			m_pixelFormatArray[1] = k32BGRAPixelFormat;
			m_pixelFormatArray[2] = k32BGRAPixelFormat;
			m_pixelFormatArray[3] = k32BGRAPixelFormat;
		} else {
			m_pixelFormatArray[0] = kYUVSPixelFormat;
			m_pixelFormatArray[1] = kYUVSPixelFormat;
			m_pixelFormatArray[2] = kYUVSPixelFormat;
			m_pixelFormatArray[3] = kYUVSPixelFormat;
		}
	} else if (mData._color == 1) { //RGB32
		m_pixelFormatArray[0] = k32BGRAPixelFormat;
		m_pixelFormatArray[1] = k24BGRPixelFormat;
		m_pixelFormatArray[2] = kYUVSPixelFormat;
		m_pixelFormatArray[3] = kYUV420PixelFormat;
	} else if (mData._color == 2) { //YUY2
		m_pixelFormatArray[0] = kYUVSPixelFormat;
		m_pixelFormatArray[1] = kYUV420PixelFormat;
		m_pixelFormatArray[2] = k24BGRPixelFormat;
		m_pixelFormatArray[3] = k32BGRAPixelFormat;
	} else if (mData._color == 3) { //YV12
		m_pixelFormatArray[0] = kYUV420PixelFormat; //kMpegYUV420CodecType
		m_pixelFormatArray[1] = kYUVSPixelFormat;
		m_pixelFormatArray[2] = k24BGRPixelFormat;
		m_pixelFormatArray[3] = k32BGRAPixelFormat;
	} else { //RGB24
		m_pixelFormatArray[0] = k24BGRPixelFormat;
		m_pixelFormatArray[1] = k32BGRAPixelFormat;
		m_pixelFormatArray[2] = kYUVSPixelFormat;
		m_pixelFormatArray[3] = kYUV420PixelFormat;
	}

	int m_iRowOrer = 1;
	for (i = 0; i < 4; i++) {
		if (mData.m_QTMovie->SetFormat(m_pixelFormatArray[i],
						mData._raw, mData.m_iMode, mData.m_fGamma, m_iRowOrer)) {
			if (m_pixelFormatArray[i] == kYUVSPixelFormat) {
				mData.pixFormat = nsVDXPixmap::kPixFormat_YUV422_YUYV;
				mData.frameSize = mData.m_QTMovie->GetDisplayWidth()
								* mData.m_QTMovie->GetDisplayHeight() * 2;
			} else if (m_pixelFormatArray[i] == kYUV420PixelFormat) {
				mData.pixFormat = nsVDXPixmap::kPixFormat_YUV420_Planar;
				mData.frameSize = mData.m_QTMovie->GetDisplayWidth()
								* mData.m_QTMovie->GetDisplayHeight() * 1.5;
			} else if (m_pixelFormatArray[i] == k32BGRAPixelFormat) {
				mData.pixFormat = nsVDXPixmap::kPixFormat_XRGB8888;
				mData.frameSize = mData.m_QTMovie->GetDisplayWidth()
								* mData.m_QTMovie->GetDisplayHeight() * 4;
			} else {
				mData.pixFormat = nsVDXPixmap::kPixFormat_RGB888;
				mData.frameSize = mData.m_QTMovie->GetDisplayWidth()
								* mData.m_QTMovie->GetDisplayHeight() * 3;
			}
			break;
		} else {
			mContext.mpCallbacks->SetError("Cannot SetFormat: %s",
				mData.m_QTMovie->GetErrorMessage());
			//delete mData.m_QTMovie;
			return;
		}
	}

	if (mData.m_QTMovie->InitializeGraphics(mData._audio, mData.strVFW, false, mData._dither) == E_FAIL) {
		// If auto mode was selected with VFW
		// and VFW failed, try one more time using a Quicktime API
		if (bAutoMode && mData.m_iMode == QTMOVIE_MODE_VFW) {
			if (mData.m_QTMovie->IsIFrameOnly())
				mData.m_iMode = QTMOVIE_MODE_DESQNC;
			else
				mData.m_iMode = QTMOVIE_MODE_ICM;
			bAutoMode = false;
			if (bAutoColor)
				mData._color = -1;
			goto SET_AUTO_MODE;
		}

		mContext.mpCallbacks->SetError("Cannot InitializeGraphics: %s",
			mData.m_QTMovie->GetErrorMessage());
		//delete mData.m_QTMovie;
		return;
	}

	//Make sure correct mode is set if using raw mode
	if (mData._raw != 0) {
		if ( (mData._raw == QTMOVIE_RAW_MODE_RGB && mData.pixFormat != nsVDXPixmap::kPixFormat_RGB888)
				|| (mData._raw == QTMOVIE_RAW_MODE_ARGB && mData.pixFormat != nsVDXPixmap::kPixFormat_XRGB8888)
				|| (mData._raw > QTMOVIE_RAW_MODE_ARGB && mData.pixFormat != nsVDXPixmap::kPixFormat_YUV422_YUYV) ) {
			//sprintf(strStatus, "\nRaw mode %s is selected but output format is not Compatible: %s\n",
			//	_raw,mData.mCodec.CodecName);
			mContext.mpCallbacks->SetError("Raw mode %s is selected but output format is not Compatible: %s",
				mData._raw,mData.mCodec.CodecName);
			//delete mData.m_QTMovie;
			return;
		}
	}

	mData.mHeader.mFrames = mData.m_QTMovie->GetVideoDuration();
	if (mData.mHeader.mFrames > 0) {
		mData.mHeader.mDepth = mData.m_QTMovie->GetDepth();
		mData.mHeader.mWidth = mData.m_QTMovie->GetDisplayWidth();
		mData.mHeader.mHeight = mData.m_QTMovie->GetDisplayHeight();
		fpsInfo fpsI = mData.m_QTMovie->GetFPS();
		mData.mHeader.mFrameRateNumerator = fpsI.numerator;
		mData.mHeader.mFrameRateDenominator = fpsI.denominator;

		mData.m_DecoderCodec = mData.m_QTMovie->QTGetDecoderInfo();
		mData.mCodec = mData.m_QTMovie->QTGetCodecInfo();

		mData.m_vFrameInfoDisplay = mData.m_QTMovie->GetDisplayFrameInfo();
		mData.m_vFrameInfoDecode = mData.m_QTMovie->GetDecodeFrameInfo();
		mData.m_vKeyframes = mData.m_QTMovie->GetKeyFrames();
	}

#ifdef QT_VFW
	//Check to make sure pixel type of mData.m_QTMovie is still the same
	//If using VFW or DShow, it may have changed
	m_pixelFormatArray[0] = mData.m_QTMovie->GetColorSpace();
	if (m_pixelFormatArray[0] == kYUVSPixelFormat)
		vi.pixel_type = VideoInfo::CS_YUY2;
	else if (m_pixelFormatArray[0] == kYUV420PixelFormat)
		vi.pixel_type = VideoInfo::CS_YV12;
	else if (m_pixelFormatArray[0] == k32BGRAPixelFormat)
		vi.pixel_type = VideoInfo::CS_BGR32;
	else if (m_pixelFormatArray[0] == k24BGRPixelFormat)
		vi.pixel_type = VideoInfo::CS_BGR24;
	else
		vi.pixel_type == VideoInfo::CS_UNKNOWN;

	if (vi.pixel_type == VideoInfo::CS_UNKNOWN) {
		delete mData.m_QTMovie;
		sprintf(strStatus, "\nCannot find a valid codec: %s\n", mCodec.CodecName);
		env->ThrowError(strStatus);
	}
#endif

	mData.mHeader.mSampleCount = 0;

	if (mData._audio && mData.m_QTMovie->HasAudio()) {
		mData.mHeader.mFormatTag = mData.m_QTMovie->GetAudioFormat();
		mData.mHeader.mFormatTag = WAVE_FORMAT_PCM;
		mData.mHeader.mChannels = mData.m_QTMovie->GetAudioNumChannels();
		mData.mHeader.mSamplesPerSec = mData.m_QTMovie->GetAudioSampleRate();
		mData.mHeader.mAvgBytesPerSec = mData.m_QTMovie->GetBytesPerFrame()
									  * mData.mHeader.mSamplesPerSec
									  * mData.mHeader.mChannels;
		mData.mHeader.mBitsPerSample = mData.m_QTMovie->GetAudioSampleSize();
		mData.mHeader.mSampleRate.mNumerator = mData.m_QTMovie->GetAudioSampleRate();
		mData.mHeader.mSampleRate.mDenominator = 1;
		mData.mHeader.mSampleCount = mData.m_QTMovie->GetAudioSampleCount();
		mData.mHeader.mBlockAlign = mData.mHeader.mChannels * mData.mHeader.mBitsPerSample / 8;

		if (mData.mHeader.mFrames == 0) {
			// This file is audio only, create blank video for it
			mData.mHeader.mFrames = (sint64)ceil(30.0 * (float)mData.mHeader.mSampleCount / mData.mHeader.mSamplesPerSec);
			mData.mHeader.mDepth = 24;
			mData.mHeader.mWidth = 320;
			mData.mHeader.mHeight = 240;
			mData.mHeader.mFrameRateNumerator = 30;
			mData.mHeader.mFrameRateDenominator = 1;
		}

	} else
		mData._audio = 0;

	// YUY2 or YV12 mode requested, open a second movie in RGB32 format for VDUb
	if (mData._color > 1 && mData.m_iMode != QTMOVIE_MODE_VFW) {
		mData.m_QTMovieRGB32 = new CQTMovieDec;
		if (mData.m_QTMovieRGB32->OpenMovie(abuf, mData.mCodec, mData._vtrack, mData._atrack) == E_FAIL) {
			//sprintf(strStatus, "\n%s\n", mData.m_QTMovie->GetErrorMessage());
			mContext.mpCallbacks->SetError("Unable to open RGB32 movie: %ls\n%s",
				szFile, mData.m_QTMovieRGB32->GetErrorMessage());
			return;
		}
		
		if (! mData.m_QTMovieRGB32->SetFormat(k32BGRAPixelFormat, mData._raw, mData.m_iMode, mData.m_fGamma, m_iRowOrer)) {
			mContext.mpCallbacks->SetError("Cannot SetFormat RGB32: %s",
				mData.m_QTMovieRGB32->GetErrorMessage());
			return;
		}
		
		if (mData.m_QTMovieRGB32->InitializeGraphics(mData._audio, mData.strVFW, false, mData._dither) == E_FAIL) {
			mContext.mpCallbacks->SetError("Cannot InitializeGraphics RGB32: %s",
				mData.m_QTMovieRGB32->GetErrorMessage());
			return;
		}
	}
}

bool VDInputFilePluginQuicktime::Append(const wchar_t *szFile)
{
	return false;
}

bool VDInputFilePluginQuicktime::PromptForOptions(VDXHWND hwndParent, IVDXInputOptions **ppOptions)
{
	VDInputFileQuicktimeOptions *p = new VDInputFileQuicktimeOptions;

	*ppOptions = p;
	if (p == NULL) return false;

	p->AddRef();

	DialogBoxParam(g_hInst, MAKEINTRESOURCE(IDD_EXTOPENOPTS_QUICKTIME),
			(HWND)hwndParent, VDInputFileQuicktimeOptions::OptionsDlgProc, (LPARAM)p);
	return true;
}

bool VDInputFilePluginQuicktime::CreateOptions(const void *buf, uint32 len, IVDXInputOptions **ppOptions)
{
	VDInputFileQuicktimeOptions *p = new VDInputFileQuicktimeOptions;

	*ppOptions = p;
	if (p == NULL) return false;

	p->AddRef();

	if (len > p->opts.len) len = p->opts.len;
		memcpy(&(p->opts), buf, len);

	return true;
}

INT_PTR APIENTRY VDInputFilePluginQuicktime::InfoDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	VDInputFilePluginQuicktime *thisPtr;
	char buf[128];
	double fps;
	char *s;
	sint64 length;
	DWORD ticks;

	switch (message)
	{
		case WM_INITDIALOG:
				SetWindowLongPtr(hDlg, DWLP_USER, lParam);

				thisPtr = (VDInputFilePluginQuicktime *)lParam;
				if (! thisPtr)
					return FALSE;

				if (! thisPtr->mData.m_QTMovie)
					return FALSE;

				length = thisPtr->mData.mHeader.mFrames;

				fps =
					(double)thisPtr->mData.mHeader.mFrameRateNumerator
						/ (double)thisPtr->mData.mHeader.mFrameRateDenominator;

				sprintf(buf, "%dx%d, %.3f fps (%ld s)",
							thisPtr->mData.mHeader.mWidth,
							thisPtr->mData.mHeader.mHeight,
							fps,
							VDRoundToLong(1000000.0/fps));

				SetDlgItemText(hDlg, IDC_VIDEO_FORMAT, buf);

				s = buf + sprintf(buf, "%I64d frames (", length);
				ticks = VDRoundToInt(1000.0*length/fps);
				ticks_to_str(s, (buf + sizeof(buf)/sizeof(buf[0])) - s, ticks);
				sprintf(s+strlen(s),".%02d)", (ticks/10)%100);
				SetDlgItemText(hDlg, IDC_VIDEO_NUMFRAMES, buf);

				if (thisPtr->mData.m_vFrameInfoDisplay.size() > 0) {
					strcpy(buf, "Unknown");

					if (thisPtr->mData.m_iMode == QTMOVIE_MODE_VFW) {
						sprintf(buf, "%s (%c%c%c%c), VFW %s",
							thisPtr->mData.m_DecoderCodec.CodecName,
							(thisPtr->mData.m_DecoderCodec.m_iFourCC & 0xFF000000) >> 24,
							(thisPtr->mData.m_DecoderCodec.m_iFourCC & 0x00FF0000) >> 16,
							(thisPtr->mData.m_DecoderCodec.m_iFourCC & 0x0000FF00) >> 8,
							(thisPtr->mData.m_DecoderCodec.m_iFourCC & 0x000000FF),
							thisPtr->mData.mCodec.strFourCC);
					} else {
						sprintf(buf, "%s (%c%c%c%c), Mode=%d",
							thisPtr->mData.m_DecoderCodec.CodecName,
							(thisPtr->mData.m_DecoderCodec.m_iFourCC & 0xFF000000) >> 24,
							(thisPtr->mData.m_DecoderCodec.m_iFourCC & 0x00FF0000) >> 16,
							(thisPtr->mData.m_DecoderCodec.m_iFourCC & 0x0000FF00) >> 8,
							(thisPtr->mData.m_DecoderCodec.m_iFourCC & 0x000000FF),
							thisPtr->mData.m_iMode);
					}

					if (thisPtr->mData.pixFormat == nsVDXPixmap::kPixFormat_RGB888)
						strcat(buf, ", RGB24");
					else if (thisPtr->mData.pixFormat == nsVDXPixmap::kPixFormat_XRGB8888)
						strcat(buf, ", RGB32");
					else if (thisPtr->mData.pixFormat == nsVDXPixmap::kPixFormat_YUV422_YUYV)
						strcat(buf, ", YUY2"); //YCbCr 4:2:2 
					else if (thisPtr->mData.pixFormat == nsVDXPixmap::kPixFormat_YUV420_Planar)
						strcat(buf, ", YV12"); //YCbCr 4:2:0 planar
					else
						strcat(buf, ", Unknown"); //YCbCr 4:2:0 planar

					SetDlgItemText(hDlg, IDC_VIDEO_COMPRESSION, buf);

					sprintf(buf, "%ld", thisPtr->mData.m_vKeyframes.size());
					SetDlgItemText(hDlg, IDC_VIDEO_NUMKEYFRAMES, buf);

					sprintf(buf, "Raw Frame Size: %d",
									thisPtr->mData.m_QTMovie->GetRawFrameSize());
					SetDlgItemText(hDlg, IDC_VIDEO_KEYFRAMESIZES, buf);

					const double seconds = (double)thisPtr->mData.mHeader.mFrames / fps;
					const double rawOverhead = (24.0 * thisPtr->mData.mHeader.mFrames);
					const double totalSize = (double)thisPtr->mData.m_QTMovie->GetVTrackSize();
					const double videoRate = (1.0 / 125.0) * totalSize / seconds;
					const double videoOverhead = totalSize > 0 ? 100.0 * rawOverhead / (rawOverhead + totalSize) : 0;
					sprintf(buf, "%.0f kbps (%.2f%% overhead)", videoRate, videoOverhead);
					SetDlgItemText(hDlg, IDC_VIDEO_DATARATE, buf);

					/*if (mData.m_vKeyframes.size()) {
						sprintf(buf, "%ld/%I64d/%ld (%I64dK)"
									,pInfo->lVideoKMinSize
									,pInfo->i64VideoKTotalSize/pInfo->lVideoKFrames
									,pInfo->lVideoKMaxSize
									,(pInfo->i64VideoKTotalSize+1023)>>10);
					else
						strcpy(buf,"(no key frames)");
					SetDlgItemText(hDlg, IDC_VIDEO_KEYFRAMESIZES, buf);*/

					/*if (pInfo->lVideoCFrames)
						sprintf(buf, "%ld/%I64d/%ld (%I64dK)"
									,pInfo->lVideoCMinSize
									,pInfo->i64VideoCTotalSize/pInfo->lVideoCFrames
									,pInfo->lVideoCMaxSize
									,(pInfo->i64VideoCTotalSize+1023)>>10);
					else
						strcpy(buf,"(no delta frames)");
					SetDlgItemText(hDlg, IDC_VIDEO_NONKEYFRAMESIZES, buf);*/
				} else {
					// Audio only file
					SetDlgItemText(hDlg, IDC_VIDEO_COMPRESSION, "Dummy Video for Audio Only File");
				}

				/*if (pInfo->bAudioFramesIndeterminate) {
						SetDlgItemText(hDlg, IDC_AUDIO_NUMFRAMES, "(indeterminate)");
						SetDlgItemText(hDlg, IDC_AUDIO_FRAMESIZES, "(indeterminate)");
						SetDlgItemText(hDlg, IDC_AUDIO_PRELOAD, "(indeterminate)");
				} else {
					if (pInfo->lAudioFrames)
						sprintf(buf, "%ld/%I64d/%ld (%I64dK)"
								,pInfo->lAudioMinSize
								,pInfo->i64AudioTotalSize/pInfo->lAudioFrames
								,pInfo->lAudioMaxSize
								,(pInfo->i64AudioTotalSize+1023)>>10);
					else
						strcpy(buf,"(no audio frames)");
					SetDlgItemText(hDlg, IDC_AUDIO_FRAMESIZES, buf);

					sprintf(buf, "%ld chunks (%.2fs preload)", pInfo->lAudioFrames,
							(double)pInfo->lAudioPreload * pWaveFormat->nBlockAlign / pWaveFormat->nAvgBytesPerSec
							);
					SetDlgItemText(hDlg, IDC_AUDIO_LAYOUT, buf);

					const double audioRate = (double)pWaveFormat->nAvgBytesPerSec * (1.0 / 125.0);
					const double rawOverhead = 24.0 * pInfo->lAudioFrames;
					const double audioOverhead = 100.0 * rawOverhead / (rawOverhead + pInfo->i64AudioTotalSize);
					sprintf(buf, "%.0f kbps (%.2f%% overhead)", audioRate, audioOverhead);
					SetDlgItemText(hDlg, IDC_AUDIO_DATARATE, buf);
				}*/

				if (thisPtr->mData.mHeader.mSampleCount > 0) {
					sprintf(buf, "%ldHz", thisPtr->mData.mHeader.mSamplesPerSec);
					SetDlgItemText(hDlg, IDC_AUDIO_SAMPLINGRATE, buf);

					if (thisPtr->mData.mHeader.mChannels == 8)
						strcpy(buf, "7.1");
					else if (thisPtr->mData.mHeader.mChannels == 6)
						strcpy(buf, "5.1");
					else if (thisPtr->mData.mHeader.mChannels > 2)
						sprintf(buf, "%d", thisPtr->mData.mHeader.mChannels);
					else
						sprintf(buf, "%d (%s)", thisPtr->mData.mHeader.mChannels, thisPtr->mData.mHeader.mChannels>1 ? "Stereo" : "Mono");
					SetDlgItemText(hDlg, IDC_AUDIO_CHANNELS, buf);

					if (thisPtr->mData.mHeader.mFormatTag == WAVE_FORMAT_PCM) {
						sprintf(buf, "%d-bit", thisPtr->mData.mHeader.mBitsPerSample);
						SetDlgItemText(hDlg, IDC_AUDIO_PRECISION, buf);
					} else
						SetDlgItemText(hDlg, IDC_AUDIO_PRECISION, "N/A");

					sint64 len = thisPtr->mData.mHeader.mSampleCount;

					char *s = buf + sprintf(buf, "%I64d samples (", len);
					//DWORD ticks = VDRoundToInt(1000.0*len*2 //Blockalign
					//	/ 1000/*pWaveFormat->nAvgBytesPerSec*/);
					DWORD ticks = VDRoundToInt(1000.0*len/thisPtr->mData.mHeader.mSamplesPerSec);
					ticks_to_str(s, (buf + sizeof(buf)/sizeof(buf[0])) - s, ticks);
					sprintf(s+strlen(s),".%02d)", (ticks/10)%100);
					SetDlgItemText(hDlg, IDC_AUDIO_LENGTH, buf);

					////////// Attempt to detect audio compression //////////

					/*if (fmt->wFormatTag == nsVDWinFormats::kWAVE_FORMAT_EXTENSIBLE) {
						const nsVDWinFormats::WaveFormatExtensible& wfe = *(const nsVDWinFormats::WaveFormatExtensible *)fmt;

						if (wfe.mGuid == nsVDWinFormats::kKSDATAFORMAT_SUBTYPE_PCM) {
							sprintf(buf, "PCM (%d bits real, chmask %x)", wfe.mBitDepth, wfe.mChannelMask);
							SetDlgItemText(hDlg, IDC_AUDIO_COMPRESSION, buf);
						} else {
							SetDlgItemText(hDlg, IDC_AUDIO_COMPRESSION, "Unknown extended format");
						}
					} else if (fmt->wFormatTag != WAVE_FORMAT_PCM) {
						// Retrieve maximum format size.

						acmMetrics(NULL, ACM_METRIC_MAX_SIZE_FORMAT, (LPVOID)&cbwfxTemp);

						// Fill out a destination wave format (PCM).

						if (pwfxTemp = (WAVEFORMATEX *)allocmem(cbwfxTemp)) {
							pwfxTemp->wFormatTag	= WAVE_FORMAT_PCM;

							// Ask ACM to fill out the details.

							if (!acmFormatSuggest(NULL, fmt, pwfxTemp, cbwfxTemp, ACM_FORMATSUGGESTF_WFORMATTAG)) {
								if (!acmStreamOpen(&has, NULL, fmt, pwfxTemp, NULL, NULL, NULL, ACM_STREAMOPENF_NONREALTIME)) {
									if (!acmDriverID((HACMOBJ)has, &hadid, 0)) {
										memset(&add, 0, sizeof add);

										add.cbStruct = sizeof add;

										if (!acmDriverDetails(hadid, &add, 0)) {
											SetDlgItemText(hDlg, IDC_AUDIO_COMPRESSION, add.szLongName);

											fSuccessful = true;
										}
									}

									acmStreamClose(has, 0);
								}
							}

							freemem(pwfxTemp);
						}

						if (!fSuccessful) {
							char buf[32];

							wsprintf(buf, "Unknown (tag %04X)", fmt->wFormatTag);
							SetDlgItemText(hDlg, IDC_AUDIO_COMPRESSION, buf);
						}
					} else */{
						// It's a PCM format...
						char fourCC[5];
						thisPtr->mData.m_QTMovie->GetAudioFourCC(fourCC);
						sprintf(buf, "PCM (Uncompressed) (%s)", fourCC);
						SetDlgItemText(hDlg, IDC_AUDIO_COMPRESSION, buf);
						//SetDlgItemText(hDlg, IDC_AUDIO_COMPRESSION, "PCM (Uncompressed)");
					}

					/*const double audioRate = (double)thisPtr->mData.mHeader.mAvgBytesPerSec * (1.0 / 125.0);
					const double rawOverhead = 24.0 * thisPtr->mData.mHeader.mSampleCount;
					const double audioOverhead = 100.0 * rawOverhead / (rawOverhead + thisPtr->mData.m_QTMovie->GetATrackSize());
					sprintf(buf, "%.0f kbps (%.2f%% overhead)", audioRate, audioOverhead);
					SetDlgItemText(hDlg, IDC_AUDIO_DATARATE, buf);*/

					//if (thisPtr->mData.mHeader.mSampleCount > 0) {
					const double audioRatePCM = (double)thisPtr->mData.mHeader.mAvgBytesPerSec * (1.0 / 125.0);
					const double seconds = (double)thisPtr->mData.mHeader.mSampleCount / thisPtr->mData.mHeader.mSamplesPerSec;
					const double rawOverhead = (24.0 * thisPtr->mData.mHeader.mSampleCount);
					const double totalSize = (double)thisPtr->mData.m_QTMovie->GetATrackSize();
					const double audioRate = (1.0 / 125.0) * totalSize / seconds;
					const double audioOverhead = totalSize > 0 ? 100.0 * rawOverhead / (rawOverhead + totalSize) : 0;
					//sprintf(buf, "%.0f kbps (%.2f%% overhead)", audioRate, audioOverhead);
					sprintf(buf, "source = %.0f kbps : PCM = %.0f kbps ", audioRate, audioRatePCM);
					SetDlgItemText(hDlg, IDC_AUDIO_DATARATE, buf);
					//}
				}

#if 0
				if (thisPtr->mData.mHeader) {
					WAVEFORMATEX *fmt = thisPtr->audioSrc->getWaveFormat();
					DWORD cbwfxTemp;
					WAVEFORMATEX *pwfxTemp;
					HACMSTREAM has;
					HACMDRIVERID hadid;
					ACMDRIVERDETAILS add;
					bool fSuccessful = false;

					sprintf(buf, "%ldHz", fmt->nSamplesPerSec);
					SetDlgItemText(hDlg, IDC_AUDIO_SAMPLINGRATE, buf);

					if (fmt->nChannels == 8)
						strcpy(buf, "7.1");
					else if (fmt->nChannels == 6)
						strcpy(buf, "5.1");
					else if (fmt->nChannels > 2)
						sprintf(buf, "%d", fmt->nChannels);
					else
						sprintf(buf, "%d (%s)", fmt->nChannels, fmt->nChannels>1 ? "Stereo" : "Mono");
					SetDlgItemText(hDlg, IDC_AUDIO_CHANNELS, buf);

					if (fmt->wFormatTag == WAVE_FORMAT_PCM) {
						sprintf(buf, "%d-bit", fmt->wBitsPerSample);
						SetDlgItemText(hDlg, IDC_AUDIO_PRECISION, buf);
					} else
						SetDlgItemText(hDlg, IDC_AUDIO_PRECISION, "N/A");

					sint64 len = thisPtr->audioSrc->getLength();
					const WAVEFORMATEX *pWaveFormat = thisPtr->audioSrc->getWaveFormat();

					char *s = buf + sprintf(buf, "%I64d samples (", len);
					DWORD ticks = VDRoundToInt(1000.0*len*pWaveFormat->nBlockAlign/pWaveFormat->nAvgBytesPerSec);
					ticks_to_str(s, (buf + sizeof(buf)/sizeof(buf[0])) - s, ticks);
					sprintf(s+strlen(s),".%02d)", (ticks/10)%100);
					SetDlgItemText(hDlg, IDC_AUDIO_LENGTH, buf);

					////////// Attempt to detect audio compression //////////

					if (fmt->wFormatTag == nsVDWinFormats::kWAVE_FORMAT_EXTENSIBLE) {
						const nsVDWinFormats::WaveFormatExtensible& wfe = *(const nsVDWinFormats::WaveFormatExtensible *)fmt;

						if (wfe.mGuid == nsVDWinFormats::kKSDATAFORMAT_SUBTYPE_PCM) {
							sprintf(buf, "PCM (%d bits real, chmask %x)", wfe.mBitDepth, wfe.mChannelMask);
							SetDlgItemText(hDlg, IDC_AUDIO_COMPRESSION, buf);
						} else {
							SetDlgItemText(hDlg, IDC_AUDIO_COMPRESSION, "Unknown extended format");
						}
					} else if (fmt->wFormatTag != WAVE_FORMAT_PCM) {
						// Retrieve maximum format size.

						acmMetrics(NULL, ACM_METRIC_MAX_SIZE_FORMAT, (LPVOID)&cbwfxTemp);

						// Fill out a destination wave format (PCM).

						if (pwfxTemp = (WAVEFORMATEX *)allocmem(cbwfxTemp)) {
							pwfxTemp->wFormatTag	= WAVE_FORMAT_PCM;

							// Ask ACM to fill out the details.

							if (!acmFormatSuggest(NULL, fmt, pwfxTemp, cbwfxTemp, ACM_FORMATSUGGESTF_WFORMATTAG)) {
								if (!acmStreamOpen(&has, NULL, fmt, pwfxTemp, NULL, NULL, NULL, ACM_STREAMOPENF_NONREALTIME)) {
									if (!acmDriverID((HACMOBJ)has, &hadid, 0)) {
										memset(&add, 0, sizeof add);

										add.cbStruct = sizeof add;

										if (!acmDriverDetails(hadid, &add, 0)) {
											SetDlgItemText(hDlg, IDC_AUDIO_COMPRESSION, add.szLongName);

											fSuccessful = true;
										}
									}

									acmStreamClose(has, 0);
								}
							}

							freemem(pwfxTemp);
						}

						if (!fSuccessful) {
							char buf[32];

							wsprintf(buf, "Unknown (tag %04X)", fmt->wFormatTag);
							SetDlgItemText(hDlg, IDC_AUDIO_COMPRESSION, buf);
						}
					} else {
						// It's a PCM format...

						SetDlgItemText(hDlg, IDC_AUDIO_COMPRESSION, "PCM (Uncompressed)");
					}
				}
			}
#endif
			//_beginthread(_InfoDlgThread, 10000, pInfo);

			//pInfo->statTimer = SetTimer(hDlg, 1, 250, NULL);

			return (TRUE);

		case WM_COMMAND:                      
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				//if (pInfo->hWndAbort == (HWND)1)
					EndDialog(hDlg, TRUE);
				//else
				//	pInfo->hWndAbort = hDlg;
				return TRUE;
			}
			break;

		case WM_DESTROY:
			//if (pInfo->statTimer) KillTimer(hDlg, pInfo->statTimer);
			break;
#if 0
		case WM_TIMER:
			{
				char buf[128];

				sprintf(buf, "%ld", pInfo->lVideoKFrames);
				SetDlgItemText(hDlg, IDC_VIDEO_NUMKEYFRAMES, buf);

				if (pInfo->lVideoKFrames)
					sprintf(buf, "%ld/%I64d/%ld (%I64dK)"
								,pInfo->lVideoKMinSize
								,pInfo->i64VideoKTotalSize/pInfo->lVideoKFrames
								,pInfo->lVideoKMaxSize
								,(pInfo->i64VideoKTotalSize+1023)>>10);
				else
					strcpy(buf,"(no key frames)");
				SetDlgItemText(hDlg, IDC_VIDEO_KEYFRAMESIZES, buf);

				if (pInfo->lVideoCFrames)
					sprintf(buf, "%ld/%I64d/%ld (%I64dK)"
								,pInfo->lVideoCMinSize
								,pInfo->i64VideoCTotalSize/pInfo->lVideoCFrames
								,pInfo->lVideoCMaxSize
								,(pInfo->i64VideoCTotalSize+1023)>>10);
				else
					strcpy(buf,"(no delta frames)");
				SetDlgItemText(hDlg, IDC_VIDEO_NONKEYFRAMESIZES, buf);

				if (thisPtr->audioSrc) {
					if (pInfo->bAudioFramesIndeterminate) {
						SetDlgItemText(hDlg, IDC_AUDIO_NUMFRAMES, "(indeterminate)");
						SetDlgItemText(hDlg, IDC_AUDIO_FRAMESIZES, "(indeterminate)");
						SetDlgItemText(hDlg, IDC_AUDIO_PRELOAD, "(indeterminate)");
					} else {

						if (pInfo->lAudioFrames)
							sprintf(buf, "%ld/%I64d/%ld (%I64dK)"
									,pInfo->lAudioMinSize
									,pInfo->i64AudioTotalSize/pInfo->lAudioFrames
									,pInfo->lAudioMaxSize
									,(pInfo->i64AudioTotalSize+1023)>>10);
						else
							strcpy(buf,"(no audio frames)");
						SetDlgItemText(hDlg, IDC_AUDIO_FRAMESIZES, buf);

						const WAVEFORMATEX *pWaveFormat = thisPtr->audioSrc->getWaveFormat();

						sprintf(buf, "%ld chunks (%.2fs preload)", pInfo->lAudioFrames,
								(double)pInfo->lAudioPreload * pWaveFormat->nBlockAlign / pWaveFormat->nAvgBytesPerSec
								);
						SetDlgItemText(hDlg, IDC_AUDIO_LAYOUT, buf);

						const double audioRate = (double)pWaveFormat->nAvgBytesPerSec * (1.0 / 125.0);
						const double rawOverhead = 24.0 * pInfo->lAudioFrames;
						const double audioOverhead = 100.0 * rawOverhead / (rawOverhead + pInfo->i64AudioTotalSize);
						sprintf(buf, "%.0f kbps (%.2f%% overhead)", audioRate, audioOverhead);
						SetDlgItemText(hDlg, IDC_AUDIO_DATARATE, buf);
					}
				}

				double totalVideoFrames = (double)pInfo->lVideoKFrames + (sint64)pInfo->lVideoCFrames;
				if (totalVideoFrames > 0) {
					VideoSourceAVI *pvs = static_cast<VideoSourceAVI *>(&*thisPtr->videoSrc);
					const double seconds = (double)pvs->getLength() / (double)pvs->getRate().asDouble();
					const double rawOverhead = (24.0 * totalVideoFrames);
					const double totalSize = (double)(pInfo->i64VideoKTotalSize + pInfo->i64VideoCTotalSize);
					const double videoRate = (1.0 / 125.0) * totalSize / seconds;
					const double videoOverhead = totalSize > 0 ? 100.0 * rawOverhead / (rawOverhead + totalSize) : 0;
					sprintf(buf, "%.0f kbps (%.2f%% overhead)", videoRate, videoOverhead);
					SetDlgItemText(hDlg, IDC_VIDEO_DATARATE, buf);
				}
			}

			/////////

			if (pInfo->hWndAbort) {
				KillTimer(hDlg, pInfo->statTimer);
				return TRUE;
			}

			break;
#endif
		case WM_USER+256:
			EndDialog(hDlg, TRUE);  
			break;
	}
	return FALSE;
}
#if (0)
	char buf[128];
	double fps;

	//static const UINT uiCtlIds[3] = {
		//IDC_VIDEO_IFRAMES, IDC_VIDEO_PFRAMES, IDC_VIDEO_BFRAMES
	//};

	switch (message) {
	case WM_INITDIALOG:
		SetWindowLongPtr(hDlg, DWLP_USER, lParam);
		//thisPtr = (VDInputFilePluginQuicktime *)lParam;

		sprintf(buf, "Quicktime Information");
		SetWindowText(hDlg, buf);

		fps = (double)fpsI.numerator /
			(double)fpsI.denominator;

		sprintf(buf, "%dx%d, %.3f fps (%ld s)",
			mData.mHeader.mWidth, mData.mHeader.mHeight,
			fps, (long)(1000000.0 / fps + 0.5));
		SetDlgItemText(hDlg, IDC_VIDEO_FORMAT, buf);

		//SetDlgItemText(hDlg, IDC_ASPECT_RATIO, "1:1");

		sprintf(buf, "%lu", mData.mHeader.mFrames);
		SetDlgItemText(hDlg, IDC_VIDEO_NUMFRAMES, buf);		

		if (m_iMode == 4) {
			sprintf(buf, "%s (%c%c%c%c), VFW %s",
				m_DecoderCodec.CodecName,
				(m_DecoderCodec.m_iFourCC & 0xFF000000) >> 24,
				(m_DecoderCodec.m_iFourCC & 0x00FF0000) >> 16,
				(m_DecoderCodec.m_iFourCC & 0x0000FF00) >> 8,
				(m_DecoderCodec.m_iFourCC & 0x000000FF),
				mCodec.strFourCC);
		} else {
			sprintf(buf, "%s (%c%c%c%c), Mode=%s",
				m_DecoderCodec.CodecName,
				(m_DecoderCodec.m_iFourCC & 0xFF000000) >> 24,
				(m_DecoderCodec.m_iFourCC & 0x00FF0000) >> 16,
				(m_DecoderCodec.m_iFourCC & 0x0000FF00) >> 8,
				(m_DecoderCodec.m_iFourCC & 0x000000FF),
				m_iMode == 0 ? "0, MoviesTask" :
					m_iMode == 2 ? "2, DecompressSeq" :
					m_iMode == 3 ? "3, ICMDecompress" : "4, VFW");
		}
		SetDlgItemText(hDlg, IDC_VIDEO_COMPRESSION, buf);

		if (m_iMode == 4) {
			sprintf(buf, "Raw Frame Size: %d",
				mData.m_QTMovie->GetRawFrameSize());
		} else {
			sprintf(buf, "Mode=%s",
				m_iMode == 0 ? "0, MoviesTask" :
					m_iMode == 2 ? "2, DecompressSeq" :
					m_iMode == 3 ? "3, ICMDecompress" : "4, VFW");
		}
		SetDlgItemText(hDlg, IDC_VIDEO_NUMKEYFRAMES, buf);

        /*sprintf(buf, "%ld / %ld / %ld", mData.mHeader.mFrames, 0, 0);
        SetDlgItemText(hDlg, IDC_VIDEO_FRAMETYPECNT, buf);

		for (i = 0; i < 3; ++i) {
			/if (pInfo.lFrameCnt[i]) {
				sprintf(buf, "%ld / %ld / %ld (%ldK)",
					pInfo.lFrameMinSize[i],
					(long)((double)pInfo.lFrameTotalSize[i] / (double)pInfo.lFrameCnt[i] + 0.5),
					pInfo.lFrameMaxSize[i],
					(pInfo.lFrameTotalSize[i] + 512) >> 10);
			} else {
				sprintf(buf,"(no %c-frames)", "IPB"[i]);
			}/

			if (i == 0) {
				sprintf(buf,"(no %c-frames)", "IPB"[i]);
			} else {
				sprintf(buf,"(no %c-frames)", "IPB"[i]);
			}

			SetDlgItemText(hDlg, uiCtlIds[i], buf);
		}

		sprintf(buf, "%ld Kbps (%ldKB/s)",
			(long)(fps * 0.008 + 0.5),
			(long)(fps / 1024.0 + 0.5));

		SetDlgItemText(hDlg, IDC_VIDEO_AVGBITRATE, buf);*/

/*
		/////////// Audio ///////////////////////////////////////

		if (thisPtr->aframes > 0) {
			pInfo.lAudioSize = (long)
				thisPtr->audio_sample_list[thisPtr->aframes - 1].stream_pos
				+ thisPtr->audio_sample_list[thisPtr->aframes - 1].size;
		}

		switch (thisPtr->iChosenAudio & AUDIOTYPE_MASK) {

		case AUDIOTYPE_MPEG: {
			MPEGAudioHeader hdr(thisPtr->audio_first_header);
			__int64 iTotalBitrate = 0;

			for (i = 0; i < thisPtr->aframes; ++i) {
				msi = &(thisPtr->audio_sample_list[i]);
				iTotalBitrate += MPEGAudioHeader(msi->header).GetBitrateKbps();
			}

			if (thisPtr->aframes > 0) {
				iTotalBitrate = (__int64)
					((double)iTotalBitrate / (double)thisPtr->aframes + 0.5);
			}

            sprintf(buf, "%ldKHz %s, %ldKbps Layer %s",
				hdr.GetSamplingRateHz() / 1000,
				szModes[hdr.IsStereo()? 1: 0],
				(long)iTotalBitrate,
				szLayers[hdr.GetLayer()]);

			SetDlgItemText(hDlg, IDC_AUDIO_FORMAT, buf);

			} break;

        case AUDIOTYPE_AC3: {
			int brate, chans;
			long srate, syncinfo[2];

			syncinfo[0] = 0x770B;
			syncinfo[1] = thisPtr->audio_first_header;

			if (ac3_syncinfo(syncinfo, &srate, &brate, &chans)) {
				sprintf(buf, "%ldKHz %s, %ldKbps AC-3",
					srate / 1000, szModes[chans - 1], brate);

				SetDlgItemText(hDlg, IDC_AUDIO_FORMAT, buf);
			}

            } break;

        case AUDIOTYPE_LPCM: {
			long srate = (thisPtr->audio_first_header & 0x30)? 96000: 48000;
			int chans = (thisPtr->audio_first_header & 7)? 2: 1;

			sprintf(buf, "%ldKHz %s, %ldKbps LPCM",
				srate / 1000, szModes[chans - 1],
				(long)(((double)srate * (double)chans * 2.0) / 125.0 + 0.5));

			SetDlgItemText(hDlg, IDC_AUDIO_FORMAT, buf);

            } break;
        }

		if (thisPtr->iChosenAudio) {
			sprintf(buf, "%ld", thisPtr->aframes);
			SetDlgItemText(hDlg, IDC_AUDIO_NUMFRAMES, buf);

			sprintf(buf, "%ldK", (pInfo.lAudioSize + 512) / 1024);
			SetDlgItemText(hDlg, IDC_AUDIO_SIZE, buf);

			sprintf(buf, "%ld ms", (long)((thisPtr->afirstPTS - thisPtr->firstPTS) / 90.0 + 0.5));
			SetDlgItemText(hDlg, IDC_AUDIO_SKEW, buf);
		}*/

		return TRUE;
	case WM_COMMAND:
		if (LOWORD(wParam) == IDOK) {
			EndDialog(hDlg, TRUE);
			return TRUE;
		}
		break;
	}

    return FALSE;
}
#endif

void VDInputFilePluginQuicktime::DisplayInfo(VDXHWND hwndParent)
{
	DialogBoxParam(g_hInst, MAKEINTRESOURCE(IDD_QUICKTIME_INFO),
		GetActiveWindow(), InfoDlgProc, (LPARAM)this);
}

bool VDInputFilePluginQuicktime::GetVideoSource(int index, IVDXVideoSource **ppVS)
{
	*ppVS = NULL;

	if (index)
		return false;

	IVDXVideoSource *pVS = new VDVideoSourcePluginQuicktime(mData);
	if (!pVS)
		return false;

	*ppVS = pVS;
	pVS->AddRef();
	return true;
}

bool VDInputFilePluginQuicktime::GetAudioSource(int index, IVDXAudioSource **ppAS)
{
	*ppAS = NULL;

	if (index || mData._audio == 0)
		return false;

	IVDXAudioSource *pAS = new VDAudioSourcePluginQuicktime(mData);
	if (!pAS)
		return false;

	*ppAS = pAS;
	pAS->AddRef();
	return true;
}

///////////////////////////////////////////////////////////////////////////////

class VDInputFileDriverPluginQuicktime : public vdxunknown<IVDXInputFileDriver> {
public:
	VDInputFileDriverPluginQuicktime(const VDInputDriverContext& context);
	~VDInputFileDriverPluginQuicktime();

	int VDXAPIENTRY AddRef();
	int VDXAPIENTRY Release();

	int		VDXAPIENTRY DetectBySignature(const void *pHeader, sint32 nHeaderSize, const void *pFooter, sint32 nFooterSize, sint64 nFileSize);
	bool	VDXAPIENTRY CreateInputFile(uint32 flags, IVDXInputFile **ppFile);

protected:
	int		mRefCount;

	const VDInputDriverContext& mContext;
};

VDInputFileDriverPluginQuicktime::VDInputFileDriverPluginQuicktime(const VDInputDriverContext& context)
	: mRefCount(0)
	, mContext(context)
{
}

VDInputFileDriverPluginQuicktime::~VDInputFileDriverPluginQuicktime()
{
}

int VDXAPIENTRY VDInputFileDriverPluginQuicktime::AddRef()
{
	return ++mRefCount;
}

int VDXAPIENTRY VDInputFileDriverPluginQuicktime::Release()
{
	int rv = --mRefCount;
	if (!rv)
		delete this;
	return rv;
}

const uint8 Quicktime_sig[] = 
{
	// ".... wide .o^"
	// 00 00 00 08 77 69 64 65 01 6F 5E E4
	0x00, 0xFF,
	0x00, 0xFF,
	0x00, 0xFF,
	0x08, 0xFF,

	0x77, 0xFF,
	0x69, 0xFF,
	0x64, 0xFF,
	0x65, 0xFF
};


const uint8 Quicktime_sig2[] = 
{
	// ".... ftyp qt  "
	0x00, 0xFF,
	0x00, 0xFF,
	0x00, 0xFF,
	0x14, 0x00, // 0x20 "... ftyp qt  "

	0x66, 0xFF,
	0x74, 0xFF,
	0x79, 0xFF,
	0x70, 0xFF,

	0x71, 0xFF,
	0x74, 0xFF,
	0x20, 0xFF,
	0x20, 0xFF
};

const uint8 Quicktime_sig3[] = 
{
	// mp4 avc1
	// ".... ftyp avc1"
	0x00, 0xFF,
	0x00, 0xFF,
	0x00, 0xFF,
	0x18, 0xFF,

	0x66, 0xFF,
	0x74, 0xFF,
	0x79, 0xFF,
	0x70, 0xFF,

	0x61, 0xFF,
	0x76, 0xFF,
	0x63, 0xFF,
	0x31, 0xFF
};

const uint8 Quicktime_sig4[] = 
{
	// mp4
	// ".... ftyp isom"
	0x00, 0xFF,
	0x00, 0xFF,
	0x00, 0xFF,
	0x14, 0xF0,

	0x66, 0xFF,
	0x74, 0xFF,
	0x79, 0xFF,
	0x70, 0xFF,

	0x69, 0xFF,
	0x73, 0xFF,
	0x6F, 0xFF,
	0x6D, 0xFF
};

const uint8 Quicktime_sig5[] = 
{
	// mp4
	// ".... ftyp mp41"
	0x00, 0xFF,
	0x00, 0xFF,
	0x00, 0xFF,
	0x1C, 0xF0,

	0x66, 0xFF,
	0x74, 0xFF,
	0x79, 0xFF,
	0x70, 0xFF,

	0x6D, 0xFF,
	0x70, 0xFF,
	0x34, 0xFF,
	0x31, 0xF0 // 0x31 or 0x32
};

const uint8 Quicktime_sig6[] = 
{
	// mov
	// "....mdat...."
	0x00, 0xFF,
	0x00, 0xFF,
	0x00, 0xFF,
	0x01, 0xFF,

	0x6D, 0xFF,
	0x64, 0xFF,
	0x61, 0xFF,
	0x74, 0xFF,

	0x00, 0xFF,
	0x00, 0xFF,
	0x00, 0xFF,
	0x00, 0xFF
};


int VDXAPIENTRY VDInputFileDriverPluginQuicktime::DetectBySignature(const void *pHeader, sint32 nHeaderSize, const void *pFooter, sint32 nFooterSize, sint64 nFileSize)
{
	return -1;
/*
	int i;
	uint8 *hdr = (uint8 *)pHeader;
	bool bFound = false;

	if (nHeaderSize >= sizeof(Quicktime_sig2) / 2) {
		for (i = 0; i < sizeof(Quicktime_sig) / 2; i++) {
			if ((Quicktime_sig[i*2] & Quicktime_sig[1+i*2])
						!= ((uint8)hdr[i] & Quicktime_sig[1+i*2])) {
				bFound = false;
				break;
			} else
				bFound = true;
		}
		if (bFound) return 1;

		for (i = 0; i < sizeof(Quicktime_sig2) / 2; i++) {
			if ((Quicktime_sig2[i*2] & Quicktime_sig2[1+i*2])
						!= ((uint8)hdr[i] & Quicktime_sig2[1+i*2])) {
				bFound = false;
				break;
			} else
				bFound = true;
		}
		if (bFound) return 1;

		for (i = 0; i < sizeof(Quicktime_sig3) / 2; i++) {
			if ((Quicktime_sig3[i*2] & Quicktime_sig3[1+i*2])
						!= ((uint8)hdr[i] & Quicktime_sig3[1+i*2])) {
				bFound = false;
				break;
			} else
				bFound = true;
		}
		if (bFound) return 1;

		for (i = 0; i < sizeof(Quicktime_sig4) / 2; i++) {
			if ((Quicktime_sig4[i*2] & Quicktime_sig4[1+i*2])
						!= ((uint8)hdr[i] & Quicktime_sig4[1+i*2])) {
				bFound = false;
				break;
			} else
				bFound = true;
		}
		if (bFound) return 1;

		for (i = 0; i < sizeof(Quicktime_sig5) / 2; i++) {
			if ((Quicktime_sig5[i*2] & Quicktime_sig5[1+i*2])
						!= ((uint8)hdr[i] & Quicktime_sig5[1+i*2])) {
				bFound = false;
				break;
			} else
				bFound = true;
		}
		if (bFound) return 1;

		for (i = 0; i < sizeof(Quicktime_sig6) / 2; i++) {
			if ((Quicktime_sig6[i*2] & Quicktime_sig6[1+i*2])
						!= ((uint8)hdr[i] & Quicktime_sig6[1+i*2])) {
				bFound = false;
				break;
			} else
				bFound = true;
		}
		if (bFound) return 1;
	}

	return -1;
*/
}

bool VDXAPIENTRY VDInputFileDriverPluginQuicktime::CreateInputFile(uint32 flags, IVDXInputFile **ppFile)
{
	IVDXInputFile *p = new VDInputFilePluginQuicktime(mContext);
	if (!p)
		return false;

	*ppFile = p;
	p->AddRef();
	return true;
}

bool VDXAPIENTRY Quicktime_create(const VDInputDriverContext *pContext, IVDXInputFileDriver **ppDriver)
{
	IVDXInputFileDriver *p = new VDInputFileDriverPluginQuicktime(*pContext);
	if (!p)
		return false;
	*ppDriver = p;
	p->AddRef();
	return true;
}

const VDInputDriverDefinition Quicktime_input = 
{
	sizeof(VDInputDriverDefinition),
	VDInputDriverDefinition::kFlagSupportsVideo
		/*| VDInputDriverDefinition::kFlagCustomSignature*/,
	0,
	6, //sizeof (Quicktime_sig),
	Quicktime_sig,
	L"*.mov|*.mp4",
	L"Quicktime Video files (*.mov;*.mp4)|*.mov;*.mp4",
	L"Quicktime Video Input",
	Quicktime_create
};

const VDPluginInfo Quicktime = 
{
	sizeof(VDPluginInfo),
	L"Quicktime",
	L"Josh Harris (tateu)",
	L"Loads Quicktime files.",
	0x01000000,
	kVDPluginType_Input,
	0,
	kVDPlugin_APIVersion,
	kVDPlugin_APIVersion,
	kVDPlugin_InputDriverAPIVersion,
	kVDPlugin_InputDriverAPIVersion,
	&Quicktime_input
};

///////////////////////////////////////////////////////////////////////////

const VDPluginInfo *const kPlugins[] = {
	&Quicktime,
	NULL
};

extern "C" __declspec(dllexport) const VDPluginInfo *const * VDXAPIENTRY VDGetPluginInfo()
{
	return kPlugins;
}
